// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_texture.h
 *  \brief      MGL テクスチャ
 *  \date       Since: December 20, 2020. 8:47:56 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TEXTURE_H_1608421676
#define INCGUARD_MGL_TEXTURE_H_1608421676

#include <mgl/render/mgl_renderer_set.h>

namespace MGL::Render
{
//! MGL テクスチャクラス
class Texture
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Texture() noexcept
        : _resource(nullptr)
        , _withKey(false)
        , _key()
    {
    }

    Texture(SharedTextureResource resource) noexcept;
    Texture(TextureKey key) noexcept;
    Texture(TextureKey key, const File::PathView &imagePath, bool isAsync, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    Texture(TextureKey key, const void *imageData, size_t dataSize, bool isAsync, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    Texture(const File::PathView &imagePath, bool isAsync, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    Texture(const void *imageData, size_t dataSize, bool isAsync, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;

    bool Load(TextureKey key) noexcept;
    bool Load(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool LoadAsync(TextureKey key, const File::PathView &imagePath, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool Load(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool LoadAsync(TextureKey key, const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;

    bool Load(const File::PathView &imagePath, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool LoadAsync(const File::PathView &imagePath, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool Load(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    bool LoadAsync(const void *imageData, size_t dataSize, TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;

    bool Create(TextureKey key, const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept;
    bool Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept;

    bool CreateRenderTarget(TextureKey key, uint32_t width, uint32_t height) noexcept;
    bool CreateRenderTarget(uint32_t width, uint32_t height) noexcept;

    [[nodiscard]] SharedTextureResource GetResource() const noexcept;

    [[nodiscard]] bool IsValid() const noexcept;
    [[nodiscard]] bool IsLoading() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型にキャストした際にテクスチャの有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    explicit operator bool() const noexcept
    {
        return IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態を否定演算子で取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    bool operator!() const noexcept
    {
        return !IsValid();
    }

    [[nodiscard]] Vector2 GetSize() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャリソースを破棄
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Destroy() noexcept
    {
        if (IsValid() && _withKey)
        {
            return Destroy(_key);
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャリソースを破棄
     *  \param[in]  key     破棄するテクスチャリソースのキー
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    static bool Destroy(TextureKey key) noexcept
    {
        return RendererSet::GetInstance().GetTextureStorage().Destroy(key);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャローダー登録用テンプレート
     *  \tparam     LoaderClass     登録するテクスチャローダーのクラス
     *  \param[in]  loaderKey       登録するテクスチャローダーのキー
     *  \param[in]  args            コンストラクタに渡す引数
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    template <class LoaderClass, class... Args>
    constexpr static bool RegisterLoader(TextureLoaderKey loaderKey, Args... args) noexcept
    {
        return RendererSet::GetInstance().GetTextureStorage().RegisterLoader(
                loaderKey,
                STL::make_unique<LoaderClass>(args...));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトのテクスチャローダーを設定
     *  \param[in]  loaderKey   設定するローダーキー
     */
    /* ------------------------------------------------------------------------- */
    static void SetDefaultLoader(TextureLoaderKey loaderKey) noexcept
    {
        RendererSet::GetInstance().GetTextureStorage().SetDefaultLoader(loaderKey);
    }

private:
    SharedTextureResource _resource;
    bool _withKey;
    TextureKey _key;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_TEXTURE_H_1608421676

// vim: et ts=4 sw=4 sts=4
