// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_renderer_set.h
 *  \brief      MGL レンダラセット
 *  \date       Since: November 28, 2020. 20:53:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RENDERER_SET_H_1606564400
#define INCGUARD_MGL_RENDERER_SET_H_1606564400

#include <mgl/initialize/mgl_initializer_delegate.h>
#include <mgl/mgl_environment.h>
#include <mgl/render/mgl_renderer_2d_delegate.h>
#include <mgl/render/mgl_texture_storage.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Render
{
//! MGL レンダラセット
class RendererSet final : public SharedSingleton<RendererSet>
{
public:
    static STL::unique_ptr<RendererSet> &GetInstanceRef() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    RendererSet() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \param[in]  initializer     イニシャライザクラス
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Initialize(const InitializerDelegate &initializer) noexcept
    {
        auto textureGenerator = initializer.MakeTextureGenerator();
        if (textureGenerator == nullptr)
        {
            return false;
        }

        _textureStorage = STL::make_unique<TextureStorage>(std::move(textureGenerator));
        if (_textureStorage == nullptr)
        {
            return false;
        }

        auto renderer2D = initializer.MakeRenderer2D();
        if (renderer2D == nullptr)
        {
            return false;
        }
        _renderer2D = std::move(renderer2D);

        initializer.MakeTextureLoader(*(_textureStorage.get()));

        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  レンダラが初期化可能かを取得
     *  \retval true    初期化可能
     *  \retval false   初期化不可能
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool CanInitializeRenderer() const noexcept
    {
        return _renderer2D->CanInitialize();
    }

    [[nodiscard]] bool InitializeRenderer() const noexcept
    {
        if (_renderer2D == nullptr)
        {
            return false;
        }

        return _renderer2D->Initialize();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  テクスチャストレージの取得
     *  \return テクスチャストレージ
     */
    /* ------------------------------------------------------------------------- */
    TextureStorage &GetTextureStorage() noexcept
    {
        return *(_textureStorage.get());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  2Dレンダラの取得
     *  \return 2Dレンダラ
     */
    /* ------------------------------------------------------------------------- */
    Renderer2DDelegate &GetRenderer2D() noexcept
    {
        return *_renderer2D;
    }

private:
    STL::unique_ptr<TextureStorage> _textureStorage;
    STL::unique_ptr<Renderer2DDelegate> _renderer2D;
};

}    // namespace MGL::Render
#endif    // INCGUARD_MGL_RENDERER_SET_H_1606564400

// vim: et ts=4 sw=4 sts=4
