// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_renderer_2d_delegate.h
 *  \brief      MGL 2Dレンダラ デリゲート
 *  \date       Since: November 28, 2020. 21:09:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RENDERER_2D_DELEGATE_H_1606565371
#define INCGUARD_MGL_RENDERER_2D_DELEGATE_H_1606565371

#include <mgl/common/mgl_color.h>
#include <mgl/math/mgl_rectangle.h>
#include <mgl/math/mgl_vector2.h>
#include <mgl/mgl_environment.h>
#include <mgl/render/mgl_draw_option_2d.h>
#include <mgl/render/mgl_texture_loader.h>

namespace MGL::Render
{
//! MGL 2Dレンダラデリゲート
class Renderer2DDelegate
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    virtual ~Renderer2DDelegate() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  レンダラの種類を取得
     *  \return レンダラの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual RendererType GetRendererType() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  レンダラが初期化可能な状態かを取得
     *  \retval true    初期化可能
     *  \retval false   初期化不可能
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool CanInitialize() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  初期化処理
     *  \retval true    成功
     *  \retval false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Initialize() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  描画の実行
     */
    /* ------------------------------------------------------------------------- */
    virtual void Flush() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      描画領域のクリア
     *  \param[in]  color   クリアする色
     */
    /* ------------------------------------------------------------------------- */
    virtual void Clear(const Color &color) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ラインの描画
     *  \param[in]  start   開始点
     *  \param[in]  end     終了点
     *  \param[in]  color   描画する色
     */
    /* ------------------------------------------------------------------------- */
    virtual void DrawLine(const Vector2 &start, const Vector2 &end, const Color &color) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      矩形の描画
     *  \param[in]  rectangle   描画する矩形
     *  \param[in]  color       描画する色
     *  \param[in]  option      描画オプション
     */
    /* ------------------------------------------------------------------------- */
    virtual void DrawRectangle(const Rectangle &rectangle, const Color &color, const DrawOption2D &option) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スプライトの描画
     *  \param[in]  textureResource     描画するテクスチャのリソース
     *  \param[in]  position            描画する位置
     *  \param[in]  sourceRectangle     描画するテクスチャの領域
     *  \param[in]  option              描画オプション
     */
    /* ------------------------------------------------------------------------- */
    virtual void DrawSprite(
        const Vector2 &position,
        SharedTextureResource textureResource,
        const Rectangle &sourceRectangle,
        const DrawOption2D &option) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットの設定
     *  \param[in]  renderTarget    設定するレンダーターゲット
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool SetRenderTarget(SharedTextureResource renderTarget) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在のレンダーターゲットを取得
     *  \return     現在のレンダーターゲット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual SharedTextureResource GetRenderTarget() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      メインレンダーターゲットを取得
     *  \return     メインレンダーターゲット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual SharedTextureResource GetMainRenderTarget() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シザーを設定
     *  \param[in]  isEnabled   有効フラグ
     *  \param[in]  rectangle   設定する矩形
     */
    /* ------------------------------------------------------------------------- */
    virtual void SetScissor(bool isEnabled, const Rectangle &rectangle) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シザーが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsEnabledScissor() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シザー矩形を取得
     *  \return     シザー矩形
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual Rectangle GetScissorRectangle() const noexcept = 0;
};
}    // namespace MGL::Render
#endif    // INCGUARD_MGL_RENDERER_2D_DELEGATE_H_1606565371

// vim: et ts=4 sw=4 sts=4
