// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_renderer_2d.h
 *  \brief      MGL 2Dレンダラ
 *  \date       Since: December 20, 2020. 11:06:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RENDERER_2D_H_1608429988
#define INCGUARD_MGL_RENDERER_2D_H_1608429988

#include <mgl/mgl_environment.h>
#include <mgl/render/mgl_renderer_2d_delegate.h>
#include <mgl/render/mgl_renderer_set.h>
#include <mgl/render/mgl_texture.h>
#include <mgl/render/mgl_texture_with_bounds.h>

namespace MGL::Render
{
//! 2Dレンダラクラス
class Renderer2D
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Renderer2D() noexcept
        : _renderer(RendererSet::GetInstance().GetRenderer2D())
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダラ本体を取得
     *  \return     レンダラ本体
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Renderer2DDelegate &GetRenderer() const noexcept
    {
        return _renderer;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダラの種類を取得
     *  \return     使用中のレンダラの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] RendererType GetRendererType() const noexcept
    {
        return _renderer.GetRendererType();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      描画領域のクリア
     *  \param[in]  color   クリアする色
     */
    /* ------------------------------------------------------------------------- */
    void Clear(const Color &color) noexcept
    {
        _renderer.Clear(color);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ラインの描画
     *  \param[in]  start   開始点
     *  \param[in]  end     終了点
     *  \param[in]  color   描画する色
     */
    /* ------------------------------------------------------------------------- */
    void DrawLine(const Vector2 &start, const Vector2 &end, const Color &color) noexcept
    {
        _renderer.DrawLine(start, end, color);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      矩形の描画
     *  \param[in]  rectangle   描画する矩形
     *  \param[in]  color       描画する色
     *  \param[in]  option      描画オプション
     */
    /* ------------------------------------------------------------------------- */
    void DrawRectangle(const Rectangle &rectangle, const Color &color, const DrawOption2D &option) noexcept
    {
        _renderer.DrawRectangle(rectangle, color, option);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      矩形の描画
     *  \param[in]  rectangle   描画する矩形
     *  \param[in]  color       描画する色
     */
    /* ------------------------------------------------------------------------- */
    void DrawRectangle(const Rectangle &rectangle, const Color &color) noexcept
    {
        DrawRectangle(rectangle, color, _option);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スプライトの描画
     *  \param[in]  texture             描画するテクスチャのリソース
     *  \param[in]  position            描画する位置
     *  \param[in]  sourceRectangle     描画するテクスチャの領域
     *  \param[in]  option              描画オプション
     */
    /* ------------------------------------------------------------------------- */
    void DrawSprite(const Vector2 &position,
                    const Texture &texture,
                    const Rectangle &sourceRectangle,
                    const DrawOption2D &option) noexcept
    {
        if (!texture.IsValid() || texture.IsLoading())
        {
            return;
        }

        _renderer.DrawSprite(position, texture.GetResource(), sourceRectangle, option);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スプライトの描画
     *  \param[in]  texture             描画するテクスチャのリソース
     *  \param[in]  position            描画する位置
     *  \param[in]  sourceRectangle     描画するテクスチャの領域
     */
    /* ------------------------------------------------------------------------- */
    void DrawSprite(const Vector2 &position,
                    const Texture &texture,
                    const Rectangle &sourceRectangle) noexcept
    {
        DrawSprite(position, texture, sourceRectangle, _option);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スプライトの描画
     *  \param[in]  position            描画する位置
     *  \param[in]  textureWithBound    描画するテクスチャと領域
     *  \param[in]  option              描画オプション
     */
    /* ------------------------------------------------------------------------- */
    void DrawSprite(const Vector2 &position,
                    const TextureWithBounds &textureWithBound,
                    const DrawOption2D &option) noexcept
    {
        DrawSprite(position, textureWithBound.GetTexture(), textureWithBound.GetBounds(), option);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スプライトの描画
     *  \param[in]  position            描画する位置
     *  \param[in]  textureWithBound    描画するテクスチャと領域
     */
    /* ------------------------------------------------------------------------- */
    void DrawSprite(const Vector2 &position, const TextureWithBounds &textureWithBound) noexcept
    {
        DrawSprite(position, textureWithBound.GetTexture(), textureWithBound.GetBounds());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットの設定
     *  \param[in]  renderTarget    設定するレンダーターゲット
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool SetRenderTarget(const Texture &renderTarget) noexcept
    {
        return _renderer.SetRenderTarget(renderTarget.GetResource());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在のレンダーターゲットを取得
     *  \return     現在のレンダーターゲット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Texture GetRenderTarget() const noexcept
    {
        return {_renderer.GetRenderTarget()};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      メインレンダーターゲットを取得
     *  \return     メインレンダーターゲット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Texture GetMainRenderTarget() const noexcept
    {
        return {_renderer.GetMainRenderTarget()};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトのレンダーターゲットに戻す
     */
    /* ------------------------------------------------------------------------- */
    void ResetRenderTarget() noexcept
    {
        SetRenderTarget(GetMainRenderTarget());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シザーの設定
     *  \param[in]  isEnabled   有効フラグ
     *  \param[in]  rectangle   シザー矩形
     */
    /* ------------------------------------------------------------------------- */
    void SetScissor(bool isEnabled, const Rectangle &rectangle = Rectangle()) noexcept
    {
        _renderer.SetScissor(isEnabled, rectangle);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      描画オプションの取得
     *  \return     描画オプション
     */
    /* ------------------------------------------------------------------------- */
    constexpr DrawOption2D &GetDrawOption() noexcept
    {
        return _option;
    }

private:
    Renderer2DDelegate &_renderer;
    DrawOption2D _option;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_RENDERER_2D_H_1608429988

// vim: et ts=4 sw=4 sts=4
