// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_render_types.h
 *  \brief      MGL 描画関連の定義
 *  \date       Since: December 8, 2020. 15:35:33 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RENDER_TYPES_H_1607409333
#define INCGUARD_MGL_RENDER_TYPES_H_1607409333

#include <cstddef>

#include <mgl/hash/mgl_hash_fnv1a.h>

namespace MGL::Render
{
//! レンダラタイプの型
enum class RendererType : uint32_t {};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列からレンダラタイプを表す値を生成
 *  \param[in]  type    レンダラタイプを表す文字列
 *  \return     文字列から生成された値
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] constexpr RendererType MakeRendererType(const char *type) noexcept
{
    return RendererType{Hash::FNV1a(type)};
}

//! ピクセルフォーマット
enum class PixelFormat : uint32_t
{
    Unknown,        //!< 不明
    RGBA8_UNorm,    //!< 32bit RGBA 各8bit
    BGRA8_UNorm,    //!< 32bit BGRA 各8bit

    B5G6R5_UNorm,    //!< 16bit BGR(5,6,5)
    A1BGR5_UNorm,    //!< 16bit ABGR(1,5,5,5)
    BGR5A1_UNorm,    //!< 16bit BGRA(5,5,5,1)
    ABGR4_UNorm,     //!< 16bit ABGR 各4bit
    BGRA4_UNorm,     //!< 16bit BGRA 各4bit

    A8_UNorm,        //!< 8bit アルファのみ

    // NOTE:
    //  全てを列挙するときりがないので，とりあえず使いそうなもののみを選定．
    //  ここに無くてもレンダラデリゲート側で使用することは可能．（アプリケーション側からはUnknownになる）

    Invalid = 0xFFFFFFFF  //!< 無効値
};

//! サンプラー
enum class SamplerType : uint8_t
{
    Invalid,    //!< 無効
    Nearest,    //!< 最近傍補間
    Linear,     //!< 線形補間
};

//! ブレンドモード
enum class BlendMode : uint8_t
{
    None,     //!< 無効
    Alpha,    //!< アルファブレンド
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ピクセルフォーマットから1ピクセルあたりのバイト数を取得
 *  \param[in]  format  ピクセルフォーマット
 *  \return     バイト数
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] constexpr size_t GetPixelPerBytes(PixelFormat format) noexcept
{
    switch (format)
    {
        case PixelFormat::RGBA8_UNorm:
        case PixelFormat::BGRA8_UNorm:
            return 4;

        case PixelFormat::B5G6R5_UNorm:
        case PixelFormat::A1BGR5_UNorm:
        case PixelFormat::BGR5A1_UNorm:
        case PixelFormat::ABGR4_UNorm:
        case PixelFormat::BGRA4_UNorm:
            return 2;

        case PixelFormat::A8_UNorm:
            return 1;

        case PixelFormat::Unknown:
        case PixelFormat::Invalid:
            return 0;
    }

    return 0;
}
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_RENDER_TYPES_H_1607409333

// vim: et ts=4 sw=4 sts=4
