// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_draw_option_2d.h
 *  \brief      MGL 2D描画オプション
 *  \date       Since: December 2, 2020. 1:59:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_DRAW_OPTION_2D_H_1606841963
#define INCGUARD_MGL_DRAW_OPTION_2D_H_1606841963

#include <mgl/common/mgl_alignment.h>
#include <mgl/common/mgl_color.h>
#include <mgl/math/mgl_vector2.h>
#include <mgl/render/mgl_render_types.h>

namespace MGL::Render
{
//! 2D描画オプション
class DrawOption2D
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr DrawOption2D() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Reset() noexcept
    {
        *this = {};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配置情報を設定
     *  \param[in]  alignment 配置情報
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetAlignment(const Alignment &alignment) noexcept { _alignment = alignment; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配置情報を取得
     *  \return     設定中の配置情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Alignment &GetAlignment() const noexcept { return _alignment; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オフセットの設定
     *  \param[in]  offset  設定するオフセット
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetOffset(const Vector2 &offset) noexcept { _offset = offset; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オフセットの取得
     *  \return     設定中のオフセット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetOffset() const noexcept { return _offset; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の設定
     *  \param[in]  scale   スケール値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScale(const Vector2 &scale) noexcept { _scale = scale; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の取得
     *  \return     設定中のスケール値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetScale() const noexcept { return _scale; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      回転角度の設定
     *  \param[in]  degreeRotate    回転角度（Degree）
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRotate(float degreeRotate) noexcept { _degreeRotate = degreeRotate; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      回転角度の取得
     *  \return     設定中の回転角度（Degree）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr float GetRotate() const noexcept { return _degreeRotate; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フリップの設定
     *  \param[in]  isHorizontalFlip    水平方向のフリップ設定
     *  \param[in]  isVerticalFlip      垂直方向のフリップ設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetFlip(bool isHorizontalFlip, bool isVerticalFlip) noexcept
    {
        _isVerticalFlip = isVerticalFlip;
        _isHorizontalFlip = isHorizontalFlip;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      垂直方向のフリップ設定の取得
     *  \return     垂直方向のフリップ設定
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsVerticalFlip() const noexcept { return _isVerticalFlip; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      水平方向のフリップ設定の取得
     *  \return     水平方向のフリップ設定
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsHorizontalFlip() const noexcept { return _isHorizontalFlip; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      回転ピボットの設定
     *  \param[in]  pivot   回転ピボット（UV座標系）
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPivot(const MGL::Vector2 &pivot) noexcept { _pivot = pivot; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      回転ピボットの取得
     *  \return     回転ピボット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const MGL::Vector2 &GetPivot() const noexcept { return _pivot; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスクカラーの設定
     *  \param[in]  color   設定するマスクカラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetMaskColor(const Color &color) noexcept { _maskColor = color; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスクカラーの取得
     *  \return     設定中のマスクカラー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Color &GetMaskColor() const noexcept { return _maskColor; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      透過値の設定
     *  \param[in]  transparency    設定する透過値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetTransparency(float transparency) noexcept { _maskColor.alpha = transparency; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      透過値の取得
     *  \return     設定中の透過値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr float GetTransparency() const noexcept { return _maskColor.alpha; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サンプラーの設定
     *  \param[in]  sampler 設定するサンプラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetSamplerType(SamplerType sampler) noexcept { _samplerType = sampler; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サンプラーの取得
     *  \return     設定中のサンプラー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr SamplerType GetSamplerType() const noexcept { return _samplerType; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ブレンドーモードの設定
     *  \param[in]  blendMode   ブレンドモード
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetBlendMode(BlendMode blendMode) noexcept { _blendMode = blendMode; }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ブレンドーモードの取得
     *  \return     ブレンドモード
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr BlendMode GetBlendMode() const noexcept { return _blendMode; }

private:
    Alignment _alignment{kAlignmentTopLeft};
    Vector2 _offset{0.0f, 0.0f};
    Vector2 _scale{1.0f, 1.0f};
    float _degreeRotate{0.0f};
    bool _isVerticalFlip{false};
    bool _isHorizontalFlip{false};
    MGL::Vector2 _pivot{0.5f, 0.5f};
    Color _maskColor{1.0f, 1.0f, 1.0f, 1.0f};
    SamplerType _samplerType{SamplerType::Nearest};
    BlendMode _blendMode{BlendMode::Alpha};
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_DRAW_OPTION_2D_H_1606841963

// vim: et ts=4 sw=4 sts=4
