// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_indexed_font_resource.h
 *  \brief      MGL インデックス化対応のフォントリソース基底クラス
 *  \date       Since: April 23, 2023. 10:47:45 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_INDEXED_FONT_RESOURCE_H_1682214465
#define INCGUARD_MGL_INDEXED_FONT_RESOURCE_H_1682214465

#include <climits>

#include <mgl/render/font/mgl_font_resource.h>

#include <mgl/render/mgl_renderer_2d.h>

namespace MGL::Render
{
//! MGL インデックス化対応のフォントリソース基底クラス
class IndexedFontResource : public FontResource
{
public:
    explicit IndexedFontResource(FontOrigin originType) noexcept;

    ~IndexedFontResource() noexcept override = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフのインデックスを取得
     *  \param[in]  character   取得するUTF-32文字
     *  \param[in]  faceType    フェイスタイプ
     *  \return     対応したグリフのインデックス．見つからない場合は SIZE_MAX
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual size_t GetIndex(char32_t character, FontFaceType faceType) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフを取得
     *  \param[in]  index       取得するグリフのインデックス
     *  \param[in]  faceType    フェイスタイプ
     *  \param[in]  option      描画オプション
     *  \return     対応するグリフ情報．見つからない場合はnullptr
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const FontGlyph *GetGlyph(size_t index, FontFaceType faceType, const FontOption &option) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した書体を保持しているかを取得
     *  \param[in]  faceType    書体タイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool HasFontFace(FontFaceType faceType) const noexcept override = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントリソースが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsValid() const noexcept override = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      改行の送りサイズを取得
     *  \return     改行の送りサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual float GetLineAdvance() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ルビのオフセット値を取得
     *  \return     ルビのオフセット値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual float GetRubyOffset() const noexcept
    {
        return 0.0f;
    }

    bool Print(
        FontWorkdata &workdata,
        const FontOption &option,
        const char *text,
        const Text::FormatArgs &formatArgs) noexcept final;

    bool Print(
        FontWorkdata &workdata,
        const FontOption &option,
        const Text::IndexedCharacter *indexedString,
        const Text::FormatArgs &formatArgs) noexcept final;

    [[nodiscard]] Text::IndexedCharacter ToIndexedCharacter(char32_t character, FontFaceType faceType) const noexcept final;
    [[nodiscard]] char32_t ToUTF32(Text::IndexedCharacter indexedCharacter, FontFaceType faceType) const noexcept final;

    [[nodiscard]] const FontGlyph *GetGlyph(char32_t character, FontFaceType faceType, const FontOption &option) const noexcept final;
    [[nodiscard]] const FontGlyph *GetGlyph(Text::IndexedCharacter character, FontFaceType faceType, const FontOption &option) const noexcept final;

private:
    // 表示用パラメータの構造体
    struct Work
    {
        Renderer2D &renderer;                            // 2Dレンダラ
        DrawOption2D drawOption;                         // 2Dレンダラの描画オプション
        Vector2 position;                                // 表示位置
        FontFaceType faceType{FontFaceType::Default};    // フェイスタイプ
        int32_t remainCount{-1};                         // 残り表示数

        // コンストラクタ
        constexpr Work(Renderer2D &renderer_) noexcept
            : renderer(renderer_)
        {
        }
    };

    const Text::IndexedCharacter *PrintLine(
        Work &work,
        const FontOption &fontOption,
        const Text::IndexedCharacter *indexedString) noexcept;

    const Text::IndexedCharacter *PrintRuby(
        Work &work,
        const FontOption &fontOption,
        const Text::IndexedCharacter *indexedString) noexcept;

    float GetLineWidth(
        const Work &work,
        const FontOption &fontOption,
        const Text::IndexedCharacter *indexedString) noexcept;

    const Text::IndexedCharacter *GetRubyWidth(
        float &baseTextWidth,
        float &rubyTextWidth,
        uint32_t &baseTextCount,
        const Text::IndexedCharacter *indexedString,
        FontFaceType faceType,
        const FontOption &fontOption) const noexcept;

    [[nodiscard]] static FontFaceType GetFontFaceType(Text::FaceTypeIndex index) noexcept;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_INDEXED_FONT_RESOURCE_H_1682214465

// vim: et ts=4 sw=4 sts=4
