// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font_storage.h
 *  \brief      MGL フォントストレージ
 *  \date       Since: May 26, 2021. 15:04:06 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FONT_STORAGE_H_1622009046
#define INCGUARD_MGL_FONT_STORAGE_H_1622009046

#include <mutex>

#include <mgl/common/mgl_singleton.h>
#include <mgl/hash/mgl_hash_fnv1a.h>
#include <mgl/render/font/mgl_font_resource.h>
#include <mgl/stl/mgl_stl_containers.h>

namespace MGL::Render
{
//! フォントリソースにアクセスするキーの型
enum class FontKey : uint32_t {};

//! フォントリソースを管理するコンテナの型
using FontResourceMap = STL::unordered_map<FontKey, SharedFontResource>;

//! フォントキーを生成する際のハッシュのシード値
constexpr uint32_t kDefaultFontKeySeed = MGL::Hash::kFNV1aDefaultValue32;    // 32bitの値なら何でもいい．衝突する場合は変更を

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントキーを生成
 *  \param[in]  key     キーの文字列
 *  \param[in]  seed    シード値（省略時はkDefaultFontKeySeed）
 *  \return     キーから生成したハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr FontKey MakeFontKey(const char *key, uint32_t seed = kDefaultFontKeySeed) noexcept
{
    return FontKey{MGL::Hash::FNV1a(key, seed)};
}

//! フォントストレージクラス
class FontStorage : public MGL::SharedSingleton<FontStorage>
{
public:
    static STL::unique_ptr<FontStorage> &GetInstanceRef() noexcept;

    FontStorage() noexcept = default;

    bool Add(FontKey key, const SharedFontResource &resource) noexcept;
    SharedFontResource Get(FontKey key) noexcept;
    bool Remove(FontKey key) noexcept;

private:
    FontResourceMap _resources;
    std::mutex _mutex;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_FONT_STORAGE_H_1622009046

// vim: et ts=4 sw=4 sts=4
