// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font_resource.h
 *  \brief      MGL フォントリソース
 *  \date       Since: May 26, 2021. 14:59:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */
#ifndef INCGUARD_MGL_FONT_RESOURCE_H_1622008768
#define INCGUARD_MGL_FONT_RESOURCE_H_1622008768

#include <mgl/math/mgl_vector2.h>
#include <mgl/render/font/mgl_font_defs.h>
#include <mgl/render/font/mgl_font_glyph.h>
#include <mgl/render/font/mgl_font_option.h>
#include <mgl/text/mgl_text_format.h>
#include <mgl/text/mgl_text_index_converter.h>

namespace MGL::Render
{
//! フォントリソースクラス
class FontResource
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr FontResource(FontOrigin originType) noexcept
        : _indexConverter(*this)
        , _originType(originType)
    {
    }

    virtual ~FontResource() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          文字の表示
     *  \param[in,out]  workdata    フォント描画のワークデータ
     *  \param[in]      option      表示オプション
     *  \param[in]      text        表示する文字列
     *  \param[in]      formatArgs  フォーマット引数
     *  \retval         true        文字列を最後まで表示した
     *  \retval         false       文字列の表示を途中で中断した
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Print(FontWorkdata &workdata,
                       const FontOption &option,
                       const char *text,
                       const Text::FormatArgs &formatArgs) noexcept
    {
        (void)workdata;
        (void)option;
        (void)text;
        (void)formatArgs;
        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          インデックス化した文字を表示
     *  \param[in,out]  workdata        フォント描画のワークデータ
     *  \param[in]      option          表示オプション
     *  \param[in]      indexedString   表示する文字列
     *  \param[in]      formatArgs      フォーマット引数
     *  \retval         true            文字列を最後まで表示した
     *  \retval         false           文字列の表示を途中で中断した
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Print(FontWorkdata &workdata,
                       const FontOption &option,
                       const Text::IndexedCharacter *indexedString,
                       const Text::FormatArgs &formatArgs) noexcept
    {
        (void)workdata;
        (void)option;
        (void)indexedString;
        (void)formatArgs;
        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字をインデックス文字に変換
     *  \param[in]  character   変換元の文字（UTF-32）
     *  \param[in]  faceType    フェイスタイプ
     *  \return     インデックス文字．失敗時はMGL::Text::kIndexedCharacterInvalid
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual Text::IndexedCharacter ToIndexedCharacter(char32_t character, FontFaceType faceType) const noexcept
    {
        (void)character;
        (void)faceType;
        return Text::kIndexedCharacterInvalid;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス文字をUTF-32の文字に変換
     *  \param[in]  indexedCharacter    インデックス文字
     *  \param[in]  faceType            フェイスタイプ
     *  \return     インデックスに対応したUTF-32エンコードの文字．失敗時はNULL文字(0x00)
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual char32_t ToUTF32(Text::IndexedCharacter indexedCharacter, FontFaceType faceType) const noexcept
    {
        (void)indexedCharacter;
        (void)faceType;
        return 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフの取得
     *  \param[in]  character   取得するグリフの文字（UTF-32）
     *  \param[in]  faceType    フェイスタイプ
     *  \param[in]  option      描画オプション
     *  \return     対応するグリフ情報．見つからない場合はnullptr
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const FontGlyph *GetGlyph(char32_t character, FontFaceType faceType, const FontOption &option) const noexcept
    {
        (void)character;
        (void)faceType;
        (void)option;
        return nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフの取得
     *  \param[in]  character   取得するグリフの文字（インデックス文字）
     *  \param[in]  faceType    フェイスタイプ
     *  \param[in]  option      描画オプション
     *  \return     対応するグリフ情報．見つからない場合はnullptr
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const FontGlyph *GetGlyph(Text::IndexedCharacter character, FontFaceType faceType, const FontOption &option) const noexcept
    {
        (void)character;
        (void)faceType;
        (void)option;
        return nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したフェイスを保持しているかを取得
     *  \param[in]  faceType    フェイスタイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool HasFontFace(FontFaceType faceType) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントリソースが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsValid() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス文字列のコンバータを取得
     *  \return     インデックス文字列のコンバータ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Text::IndexConverter &GetIndexConverter() const noexcept
    {
        return _indexConverter;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォント機能の有効状態を取得
     *  \param[in]  feature フォントの機能
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabled(FontFeature feature) const noexcept
    {
        return (_featureFlags & static_cast<uint32_t>(feature)) != 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォントの原点タイプを取得
     *  \return     原点タイプ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr FontOrigin GetOriginType() const noexcept
    {
        return _originType;
    }

protected:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォント機能の有効状態を設定
     *  \param[in]  feature     フォントの機能
     *  \param[in]  isEnabled   設定する有効フラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetEnabled(FontFeature feature, bool isEnabled) noexcept
    {
        if (isEnabled)
        {
            _featureFlags |= static_cast<uint32_t>(feature);
        }
        else
        {
            _featureFlags &= ~(static_cast<uint32_t>(feature));
        }
    }

private:
    //! フォントリソースクラス用のインデックス文字列コンバータ
    class IndexConverter : public Text::IndexConverter
    {
    public:
        /* ------------------------------------------------------------------------- */
        /*!
         *  \brief      コンストラクタ
         */
        /* ------------------------------------------------------------------------- */
        constexpr IndexConverter(const FontResource &fontResource) noexcept
            : _resource(fontResource)
        {
        }

        /* ------------------------------------------------------------------------- */
        /*!
         *  \brief      文字をインデックス文字に変換
         *  \param[in]  character   変換元の文字（UTF-32）
         *  \param[in]  faceType    フェイスタイプ
         *  \return     インデックス文字
         */
        /* ------------------------------------------------------------------------- */
        [[nodiscard]] Text::IndexedCharacter ToIndexedCharacter(char32_t character, FontFaceType faceType) const noexcept override
        {
            return _resource.ToIndexedCharacter(character, faceType);
        }

        /* ------------------------------------------------------------------------- */
        /*!
          *  \brief      インデックス文字をUTF-32の文字に変換
          *  \param[in]  indexedCharacter   インデックス文字
          *  \param[in]  faceType           フェイスタイプ
          *  \return     インデックスに対応したUTF-32エンコードの文字
          */
        /* ------------------------------------------------------------------------- */
        [[nodiscard]] char32_t ToUTF32(Text::IndexedCharacter indexedCharacter, FontFaceType faceType) const noexcept override
        {
            return _resource.ToUTF32(indexedCharacter, faceType);
        }

        /* ------------------------------------------------------------------------- */
        /*!
         *  \brief      指定したフェイスを保持しているかを取得
         *  \param[in]  faceType    フェイスタイプ
         *  \retval     true        保持している
         *  \retval     false       保持していない
         */
        /* ------------------------------------------------------------------------- */
        [[nodiscard]] bool HasFontFace(FontFaceType faceType) const noexcept override
        {
            return _resource.HasFontFace(faceType);
        }

    private:
        const FontResource &_resource;
    };

    IndexConverter _indexConverter;
    FontOrigin _originType;
    uint32_t _featureFlags{0};
};

//! フォントリソースの共有ポインタ
using SharedFontResource = std::shared_ptr<FontResource>;

}    // namespace MGL::Render

#endif    // INCGUARD_MGL_FONT_RESOURCE_H_1622008768

// vim: et ts=4 sw=4 sts=4
