// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font_defs.h
 *  \brief      フォント関連定義
 *  \date       Since: April 29, 2023. 19:37:40 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FONT_DEFS_H_1682764660
#define INCGUARD_MGL_FONT_DEFS_H_1682764660

#include <mgl/hash/mgl_hash_fnv1a.h>
#include <mgl/math/mgl_vector2.h>

namespace MGL::Render
{
//! フォントのフェイスタイプ
enum class FontFaceType : uint32_t
{
    Default = Hash::FNV1a("default"),    //!< デフォルト
    Bold    = Hash::FNV1a("bold"),       //!< ボールド体
    Ruby    = Hash::FNV1a("ruby"),       //!< ルビ
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列からフォントのフェイスタイプを生成
 *  \param[in]  name    フェイスタイプの名前
 *  \return     フェイスタイプ
 */
/* ------------------------------------------------------------------------- */
constexpr FontFaceType MakeFontFaceType(const char *name) noexcept
{
    return FontFaceType{Hash::FNV1a(name)};
}

//! フォントの機能
enum class FontFeature : uint32_t
{
    IndexedCharacter    = 1u << 0u,     //!< インデックス文字
    GetGlyph            = 1u << 1u,     //!< グリフの取得
    Tag                 = 1u << 2u,     //!< タグ
    Ruby                = 1u << 3u,     //!< ルビ

    UnusedMask          = 0xFFFFFFF0,   //!< 未使用ビットフィールドのマスク値
};

//! フォントの原点
enum class FontOrigin : uint8_t
{
    TopLeft,        //!< 左上
    BaseLine,       //!< ベースライン
};

//! フォント描画のワークデータ
struct FontWorkdata
{
    Vector2 position;   //!< 表示位置
    int32_t limitCount{-1}; //!< 表示上限数
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_FONT_DEFS_H_1682764660

// vim: et ts=4 sw=4 sts=4
