// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font.h
 *  \brief      MGL フォント
 *  \date       Since: May 27, 2021. 16:19:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FONT_H_1622099992
#define INCGUARD_MGL_FONT_H_1622099992

#include <mgl/render/font/mgl_font_storage.h>
#include <mgl/render/mgl_renderer_set.h>
#include <mgl/text/mgl_text_format.h>

namespace MGL::Render
{
//! MGL フォントクラス
class Font
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  key     フォントキー
     */
    /* ------------------------------------------------------------------------- */
    Font(FontKey key) noexcept
        : _resource(FontStorage::GetInstance().Get(key))
        , _work()
        , _key(key)
        , _isOnceLimit(false)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Font() noexcept
        : _resource(nullptr)
        , _work()
        , _key()
        , _isOnceLimit(false)
    {
    }

    static Font AddStorage(FontKey key, const SharedFontResource &resource) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォント生成用テンプレート
     *  \tparam     FontResourceClass   追加するフォントリソースのクラス
     *  \param[in]  key                 関連付けるフォントキー
     *  \param[in]  args                コンストラクタに渡す引数
     *  \return     フォントクラス
     */
    /* ------------------------------------------------------------------------- */
    template <class FontResourceClass, class... Args>
    static Font Create(FontKey key, Args... args) noexcept
    {
        return AddStorage(key, STL::make_shared<FontResourceClass>(args...));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォントの削除
     */
    /* ------------------------------------------------------------------------- */
    static void Remove(FontKey key) noexcept
    {
        FontStorage::GetInstance().Remove(key);
    }

    bool Load(FontKey key) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントのキーを取得
     *  \return     フォントキー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr FontKey GetFontKey() const noexcept
    {
        return _key;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示位置の設定
     *  \param[in]  position    設定する表示位置
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPosition(const Vector2 &position) noexcept
    {
        _work.position = _option.firstPosition = position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示位置の設定
     *  \param[in]  x   X座標
     *  \param[in]  y   Y座標
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPosition(float x, float y) noexcept
    {
        _work.position.x = x;
        _work.position.y = y;
        _option.firstPosition = _work.position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示位置の取得
     *  \return     現在の表示位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetPosition() const noexcept
    {
        return _work.position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の設定
     *  \param[in]  scale   設定するスケール値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScale(const Vector2 &scale) noexcept
    {
        _option.scale = scale;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の設定
     *  \param[in]  scaleX  X方向のスケール値
     *  \param[in]  scaleY  Y方向のスケール値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScale(float scaleX, float scaleY) noexcept
    {
        _option.scale.x = scaleX;
        _option.scale.y = scaleY;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の設定
     *  \param[in]  scale   スケール値（X,Y共通）
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetScale(float scale) noexcept
    {
        _option.scale.x = _option.scale.y = scale;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スケール値の取得
     *  \return     現在のスケール値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetScale() const noexcept
    {
        return _option.scale;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      水平方向の配置情報の設定
     *  \param[in]  alignment  設定する配置情報
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetHorizontalAlignment(Alignment::Horizontal alignment) noexcept
    {
        _option.horizontalAlignment = alignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      水平方向の配置情報の取得
     *  \return     現在の配置情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Alignment::Horizontal GetHorizontalAlignment() const noexcept
    {
        return _option.horizontalAlignment;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスクカラーの設定
     *  \param[in]  color   設定するマスクカラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetMaskColor(const Color &color) noexcept
    {
        _option.maskColor = color;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスクカラーの取得
     *  \return     現在のマスクカラー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Color &GetMaskColor() const noexcept
    {
        return _option.maskColor;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サンプラータイプの設定
     *  \param[in]  samplerType     設定するサンプラータイプ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetSamplerType(SamplerType samplerType) noexcept
    {
        _option.samplerType = samplerType;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サンプラータイプの取得
     *  \return     現在のサンプラータイプ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr SamplerType GetSamplerType() const noexcept
    {
        return _option.samplerType;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      透過値の設定
     *  \param[in]  transparency    透過値(0.0fから1.0f)
     *  \note       マスクカラーのアルファ値に書き込む
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetTransparency(float transparency) noexcept
    {
        _option.maskColor.alpha = transparency;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      透過値の取得
     *  \return     現在の透過値
     *  \note       マスクカラーのアルファ値を返す
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr float GetTransparency() const noexcept
    {
        return _option.maskColor.alpha;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      字間と行間の設定
     *  \param[in]  margin  設定する字間と行間
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetMargin(const Vector2 &margin) noexcept
    {
        _option.margin = margin;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      字間と行間の取得
     *  \return     現在の字間と行間
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetMargin() const noexcept
    {
        return _option.margin;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォントの有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsValid() const noexcept
    {
        return _resource->IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型にキャストした際に有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    explicit operator bool() const noexcept
    {
        return IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態を否定演算子で取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    bool operator!() const noexcept
    {
        return !static_cast<bool>(*this);
    }

    bool Print(const char *text, const Text::FormatArgs &args = Text::FormatArgs()) noexcept;
    bool Print(const Text::IndexedCharacter *indexedString, const Text::FormatArgs &args = Text::FormatArgs()) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字の表示
     *  \param[in]  text    表示する文字
     *  \param[in]  args    フォーマット引数
     */
    /* ------------------------------------------------------------------------- */
    template <class... Args>
    constexpr bool Print(const char *text, const Args &...args) noexcept
    {
        return Print(text, {args...});
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス化した文字の表示
     *  \param[in]  indexedString   表示する文字
     *  \param[in]  args            フォーマット引数
     */
    /* ------------------------------------------------------------------------- */
    template <class... Args>
    constexpr bool Print(const Text::IndexedCharacter *indexedString, const Args &...args) noexcept
    {
        return Print(indexedString, {args...});
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントが扱える機能を取得
     *  \param[in]  feature     取得する機能
     *  \retval     true        扱える
     *  \retval     false       扱えない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsEnabled(FontFeature feature) const noexcept
    {
        return (_resource != nullptr) ? _resource->IsEnabled(feature) : false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したフェイスを保持しているかを取得
     *  \param[in]  faceType    フェイスタイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool HasFontFace(FontFaceType faceType) const noexcept
    {
        return (_resource != nullptr) ? _resource->HasFontFace(faceType) : false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字列をこのフォント用のインデックス文字列に変換
     *  \param[in]  text            変換する文字列
     *  \param[in]  enableFormat    テキスト整形を有効にするかのフラグ
     *  \param[in]  enableTag       タグを有効にするかのフラグ
     *  \return     インデックス文字列
     */
    /* ------------------------------------------------------------------------- */
    STL::unique_ptr<Text::IndexedCharacter[]> ToIndexedString(const char *text, bool enableFormat, bool enableTag) const noexcept
    {
        return (_resource != nullptr) ? _resource->GetIndexConverter().ToIndexedString(text, enableFormat, enableTag) : nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      UTF-32文字をこのフォント用のインデックス文字に変換
     *  \param[in]  character       変換元の文字（UTF-32）
     *  \param[in]  faceType        フェイスタイプ
     *  \return     インデックス文字
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Text::IndexedCharacter ToIndexedCharacter(char32_t character, FontFaceType faceType = FontFaceType::Default) const noexcept
    {
        return (_resource != nullptr) ? _resource->ToIndexedCharacter(character, faceType) : Text::kIndexedCharacterInvalid;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォントの原点タイプを取得
     *  \return     原点タイプ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR FontOrigin GetOriginType() const noexcept
    {
        return (_resource != nullptr) ? _resource->GetOriginType() : FontOrigin::TopLeft;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフ情報の取得
     *  \param[in]  character   UTF-32文字
     *  \param[in]  faceType    書体
     *  \return     引数に対応するグリフ情報．見つからない場合はnullptr
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR const FontGlyph *GetGlyph(char32_t character, FontFaceType faceType = FontFaceType::Default) const noexcept
    {
        return (_resource != nullptr) ? _resource->GetGlyph(character, faceType, _option) : nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      グリフ情報の取得
     *  \param[in]  character   インデックス文字
     *  \param[in]  faceType    書体
     *  \return     引数に対応するグリフ情報．見つからない場合はnullptr
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR const FontGlyph *GetGlyph(Text::IndexedCharacter character, FontFaceType faceType = FontFaceType::Default) const noexcept
    {
        return (_resource != nullptr) ? _resource->GetGlyph(character, faceType, _option) : nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示上限数の設定
     *  \param[in]  limitCount  表示上限数
     *  \param[in]  isOnce      直後のPrint()にのみ適用するかのフラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetLimits(int32_t limitCount, bool isOnce) noexcept
    {
        _work.limitCount = limitCount;
        _isOnceLimit = isOnce;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示上限数をクリア
     */
    /* ------------------------------------------------------------------------- */
    constexpr void ClearLimits() noexcept
    {
        _work.limitCount = -1;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      表示上限数の取得
     *  \return     表示上限数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr int32_t GetLimits() const noexcept
    {
        return _work.limitCount;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス文字列の整形
     *  \param[in]  text    インデックス文字列
     *  \param[in]  args    フォーマット引数
     *  \return     整形後のインデックス文字列の配列
     */
    /* ------------------------------------------------------------------------- */
    STL::vector<Text::IndexedCharacter> Format(const Text::IndexedCharacter *text, const Text::FormatArgs &args) const noexcept
    {
        return (_resource != nullptr) ? Text::Format(text, _resource->GetIndexConverter(), args) : STL::vector<Text::IndexedCharacter>();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス文字列の整形
     *  \param[in]  text    インデックス文字列
     *  \param[in]  args    フォーマット引数
     *  \return     整形後のインデックス文字列の配列
     */
    /* ------------------------------------------------------------------------- */
    template <class... Args>
    STL::vector<Text::IndexedCharacter> Format(const Text::IndexedCharacter *text, const Args &...args) noexcept
    {
        return Format(text, {args...});
    }

private:
    SharedFontResource _resource;
    FontWorkdata _work;
    FontOption _option;
    FontKey _key;
    bool _isOnceLimit;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_FONT_H_1622099992

// vim: et ts=4 sw=4 sts=4
