// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ascii_font.h
 *  \brief      MGL ASCIIフォント
 *  \date       Since: May 26, 2021. 18:36:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ASCII_FONT_H_1622021769
#define INCGUARD_MGL_ASCII_FONT_H_1622021769

#include <mgl/file/mgl_file_path_view.h>
#include <mgl/render/font/mgl_indexed_font_resource.h>
#include <mgl/render/mgl_renderer_2d.h>

namespace MGL::Render
{
//! ASCIIフォントクラス
class AsciiFont : public IndexedFontResource
{
public:
    AsciiFont(TextureKey textureKey, uint32_t characterWidth, uint32_t characterHeight, const Vector2 &sourceOffset = MGL::Vector2()) noexcept;
    AsciiFont(const TextureWithBounds &textureBounds) noexcept;
    AsciiFont(
        const File::PathView &imagePath,
        uint32_t characterWidth,
        uint32_t characterHeight,
        const Vector2 &sourceOffset = MGL::Vector2(),
        TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;
    AsciiFont(
        const void *imageData,
        size_t dataSize,
        uint32_t characterWidth,
        uint32_t characterHeight,
        const Vector2 &sourceOffset = MGL::Vector2(),
        TextureLoaderKey loaderKey = TextureStorage::kDefaultLoaderKey) noexcept;

    void Initialize() noexcept;

    [[nodiscard]] size_t GetIndex(char32_t character, FontFaceType faceType) const noexcept override;
    [[nodiscard]] const FontGlyph *GetGlyph(size_t index, FontFaceType faceType, const FontOption &option) const noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したフェイスを保持しているかを取得
     *  \param[in]  faceType    フェイスタイプ
     *  \retval     true        保持している
     *  \retval     false       保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool HasFontFace(FontFaceType faceType) const noexcept override
    {
        return (faceType == FontFaceType::Default);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このフォントリソースが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsValid() const noexcept override
    {
        return _texture.IsValid();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      改行の送りサイズを取得
     *  \return     改行の送りサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetLineAdvance() const noexcept override
    {
        return _characterSize.y;
    }

private:
    Texture _texture;
    Vector2 _characterSize;
    Vector2 _sourceOffset;
    STL::unique_ptr<FontGlyph[]> _glyphs;
};
}    // namespace MGL::Render

#endif    // INCGUARD_MGL_ASCII_FONT_H_1622021769

// vim: et ts=4 sw=4 sts=4
