// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_texture_resource.h
 *  \brief      MGL Direct3D11用テクスチャリソース
 *  \date       Since: March 1, 2021. 18:53:21 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_D3D11_TEXTURE_RESOURCE_H_1614592401
#define INCGUARD_MGL_D3D11_TEXTURE_RESOURCE_H_1614592401

#include <mgl/mgl_environment.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <mgl/render/mgl_texture_resource.h>
#include <mgl/render/d3d11/mgl_d3d11_defs.h>
#include <d3d11_1.h>
#include <atlbase.h>

namespace MGL::Render::D3D11
{
//! MGL Direct3D11用テクスチャリソース
class TextureResource : public MGL::Render::TextureResource
{
public:
    TextureResource(bool isMainRenderTarget = false) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リソースがどのレンダラ用のものかを返す
     *  \return     レンダラの種類
     */
    /* ------------------------------------------------------------------------- */
    RendererType GetRendererType() const noexcept override
    {
        return kRendererTypeDirect3D11;
    }
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態の取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    bool IsValid() const noexcept override
    {
        return _isMainRenderTarget || (_texture != nullptr);
    }
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットに設定可能かを取得
     *  \retval     true    設定可能
     *  \retval     false   不可能
     */
    /* ------------------------------------------------------------------------- */
    bool IsRenderTarget() const noexcept override
    {
        return _isMainRenderTarget || (_renderTargetView != nullptr);
    }
    
    bool Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept override;
    bool CreateRenderTarget(uint32_t width, uint32_t height) noexcept override;
    void Destroy() noexcept override;

    Vector2 GetSize() const noexcept override;
    PixelFormat GetPixelFormat() const noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      レンダーターゲットビューの取得
     *  \return     レンダーターゲットビュー
     */
    /* ------------------------------------------------------------------------- */
    ID3D11RenderTargetView *GetRenderTargetView() noexcept
    {
        return _renderTargetView.p;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シェーダリソースビューの取得
     *  \return     シェーダリソースビュー
     */
    /* ------------------------------------------------------------------------- */
    ID3D11ShaderResourceView *GetShaderResourceView() noexcept
    {
        return _shaderResourceView.p;
    }

    ID3D11Buffer *GetOrthogonalMatrixBuffer() const noexcept;

private:
    [[nodiscard]] static DXGI_FORMAT GetDXPixelFormat(PixelFormat pixelFormat) noexcept;
    [[nodiscard]] static PixelFormat GetPixelFormat(DXGI_FORMAT dxPixelFormat) noexcept;

    bool _isMainRenderTarget;

    CComPtr<ID3D11Texture2D>            _texture;
    CComPtr<ID3D11ShaderResourceView>   _shaderResourceView;
    CComPtr<ID3D11RenderTargetView>     _renderTargetView;
    CComPtr<ID3D11Buffer>               _orthogonalMatrixBuffer;

    uint32_t    _width;
    uint32_t    _height;

    PixelFormat _pixelFormat;
};
}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
#endif	// INCGUARD_MGL_D3D11_TEXTURE_RESOURCE_H_1614592401

// vim: et ts=4 sw=4 sts=4
