// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_sprite_renderer.h
 *  \brief      MGL Direct3D11用スプライトレンダラ
 *  \date       Since: March 6, 2021. 15:44:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_D3D11_SPRITE_RENDERER_H_1615013087
#define INCGUARD_MGL_D3D11_SPRITE_RENDERER_H_1615013087

#include <mgl/mgl_environment.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <array>
#include <atlbase.h>
#include <d3d11_1.h>
#include <mgl/render/d3d11/mgl_d3d11_object_renderer.h>
#include <mgl/render/mgl_texture_resource.h>
#include <vector>

namespace MGL::Render::D3D11
{
//! Direct3D11用スプライトレンダラ
class SpriteRenderer : public ObjectRenderer
{
public:
    //! スプライト用アトリビュート
    struct Attribute
    {
        float offset[2];        //!< オフセット（表示位置）
        float scale[2];         //!< スケール値
        float textureCoord[4];  //!< テクスチャ座標
        float color[4];         //!< マスクカラー
        float rotate;           //!< 回転角度
        float pivot[2];         //!< 回転ピボット
        uint32_t flip;
    };
    static constexpr size_t kAttributeMax = 1024;   //!< アトリビュートの最大数

    SpriteRenderer() noexcept;

    bool Initialize() noexcept override;
    void Flush(const FlushParameter &flushParameter) noexcept override;

    void Add(const Attribute &attribute, const FlushParameter &flushParameter, SharedTextureResource &texture, SamplerType samplerType, bool isEnabledBlending) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アトリビュートのストックが上限に達しているかを取得
     *  \retval     true    上限に達している
     *  \retval     false   空きがある
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsFull() const noexcept
    {
        return _attributeCount >= kAttributeMax;
    }

private:
    bool InitializeShader() noexcept;
    bool InitializeVertexBuffer() noexcept;
    bool InitializeSamplerState() noexcept;
    
    ID3D11VertexShader *GetVertexShader(size_t size) noexcept;
    ID3D11SamplerState *GetSamplerState(SamplerType samplerType) noexcept;

    bool _isAvailable;

    std::array<Attribute, kAttributeMax> _attributes;
    size_t _attributeCount;

    CComPtr<ID3D11Buffer> _vertexBuffer;

    struct VertexShader
    {
        size_t size;
        CComPtr<ID3D11VertexShader> shader;
    };
    std::vector<VertexShader> _vertexShaderArray;
    CComPtr<ID3D11PixelShader> _pixelShader;
    CComPtr<ID3D11InputLayout> _inputLayout;

    enum
    {
        kSamplerStateNearest,
        kSamplerStateLinear,
        kSamplerStateMax
    };
    CComPtr<ID3D11SamplerState> _samplers[kSamplerStateMax];

    SharedTextureResource _currentTexture;
    SamplerType _currentSamplerType;
    bool _isEnabledBlending;
};
}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
#endif	// INCGUARD_MGL_D3D11_SPRITE_RENDERER_H_1615013087

// vim: et ts=4 sw=4 sts=4
