// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_rectangle_renderer.h
 *  \brief      MGL Direct3D11用矩形レンダラ
 *  \date       Since: March 5, 2021. 17:18:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_D3D11_RECTANGLE_RENDERER_H_1614932332
#define INCGUARD_MGL_D3D11_RECTANGLE_RENDERER_H_1614932332

#include <mgl/mgl_environment.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <array>
#include <atlbase.h>
#include <d3d11_1.h>
#include <mgl/render/d3d11/mgl_d3d11_object_renderer.h>
#include <vector>

namespace MGL::Render::D3D11
{
//! Direct3D11用矩形レンダラ
class RectangleRenderer : public ObjectRenderer
{
public:
    //! 矩形用アトリビュート
    struct Attribute
    {
        float offset[4];    //!< オフセット（表示位置）
        float scale[4];     //!< スケール値
        float color[4];     //!< 色
        float rotate;       //!< 回転角度
        float pivot[2];     //!< 回転ピボット
        float unused;       //!< 未使用
    };
    static constexpr size_t kAttributeMax = 1024;       //!< アトリビュートの最大数
    
    RectangleRenderer() noexcept;

    virtual bool Initialize() noexcept override;
    virtual void Flush(const FlushParameter &flushParameter) noexcept override;

    void Add(const Attribute &attribute, const FlushParameter &flushParameter) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アトリビュートのストックが上限に達しているかを取得
     *  \retval     true    上限に達している
     *  \retval     false   空きがある
     */
    /* ------------------------------------------------------------------------- */
    bool IsFull() const noexcept
    {
        return _attributeCount >= kAttributeMax;
    }

private:
    bool InitializeShader() noexcept;
    bool InitializeVertexBuffer() noexcept;
    ID3D11VertexShader *GetVertexShader(size_t size) noexcept;

    bool _isAvailable;

    std::array<Attribute, kAttributeMax> _attributes;
    size_t _attributeCount;

    struct VertexShader
    {
        size_t size;
        CComPtr<ID3D11VertexShader> shader;
    };
    std::vector<VertexShader> _vertexShaderArray;
    CComPtr<ID3D11PixelShader> _pixelShader;
    CComPtr<ID3D11InputLayout> _inputLayout;
    CComPtr<ID3D11Buffer> _vertexBuffer;
};
}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
#endif	// INCGUARD_MGL_D3D11_RECTANGLE_RENDERER_H_1614932332

// vim: et ts=4 sw=4 sts=4
