// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_line_renderer.h
 *  \brief      MGL Direct3D11用ラインレンダラ
 *  \date       Since: April 3, 2021. 14:57:18 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_D3D11_LINE_RENDERER_H_1617429438
#define INCGUARD_MGL_D3D11_LINE_RENDERER_H_1617429438

#include <mgl/mgl_environment.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <array>
#include <atlbase.h>
#include <d3d11_1.h>
#include <mgl/render/d3d11/mgl_d3d11_object_renderer.h>
#include <vector>
#include <mgl/math/mgl_vector2.h>
#include <mgl/common/mgl_color.h>

namespace MGL::Render::D3D11
{
//! Direct3D11用ラインレンダラ
class LineRenderer : public ObjectRenderer
{
public:
    //! ライン用アトリビュート
    struct Attribute
    {
        float position[2];  //!< 座標
        float color[4];     //!< 色
    };
    static constexpr size_t kAttributeMax = 1024;   //!< アトリビュートの最大数

    LineRenderer() noexcept;

    virtual bool Initialize() noexcept override;
    virtual void Flush(const FlushParameter &flushParameter) noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アトリビュートのストックが上限に達しているかを取得
     *  \retval     true    上限に達している
     *  \retval     false   空きがある
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool IsFull() const noexcept
    {
        return _attributeCount >= (kAttributeMax - 1);
    }

    void Add(const Vector2 &start, const Vector2 &end, const Color &color, const FlushParameter &flushParameter) noexcept;
    
private:
    bool InitializeShader() noexcept;
    bool InitializeVertexBuffer() noexcept;

    bool _isAvailable;
    
    std::array<Attribute, kAttributeMax> _attributes;
    size_t _attributeCount;

    CComPtr<ID3D11VertexShader> _vertexShader;
    CComPtr<ID3D11PixelShader> _pixelShader;
    CComPtr<ID3D11InputLayout> _inputLayout;
    CComPtr<ID3D11Buffer> _vertexBuffer;
};
}   // MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
#endif	// INCGUARD_MGL_D3D11_LINE_RENDERER_H_1617429438

// vim: et ts=4 sw=4 sts=4
