// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_device.h
 *  \brief      MGL Direct3D11 デバイス管理
 *  \date       Since: March 1, 2021. 15:03:39 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_D3D11_DEVICE_H_1614578619
#define INCGUARD_MGL_D3D11_DEVICE_H_1614578619

#include <mgl/mgl_environment.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <d3d11_1.h>
#include <mgl/common/mgl_color.h>
#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/math/mgl_vector2.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <vector>
#include <atlbase.h>

namespace MGL::Render::D3D11
{
//! MGL Direct3D11 デバイス管理
class Device final : public SharedSingleton<Device>
{
public:
    static STL::unique_ptr<Device> &GetInstanceRef() noexcept;

    Device() noexcept;
    ~Device() noexcept;

    bool Initialize() noexcept;

    bool SetFrameLatency(uint32_t latency) noexcept;
    bool SetFullscreen(bool isEnable) noexcept;
    bool IsFullscreen() const noexcept;

    void ClearRenderTarget(const Color &color) noexcept;
    void Present() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デバイスコンテキストの取得
     *  \return     デバイスコンテキスト
     */
    /* ------------------------------------------------------------------------- */
    ID3D11DeviceContext *GetContext() noexcept
    {
        return _immediateContext;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スワップチェインの取得
     *  \return     スワップチェイン
     */
    /* ------------------------------------------------------------------------- */
    IDXGISwapChain *GetSwapChain() noexcept
    {
        return _swapChain;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      Direct3Dデバイスの取得
     *  \return     Direct3Dデバイス
     */
    /* ------------------------------------------------------------------------- */
    ID3D11Device *GetD3DDevice() noexcept
    {
        return _d3dDevice;
    }

    //! 動的バッファ
    struct DynamicBuffer
    {
        CComPtr<ID3D11Buffer> buffer;   //!< バッファの先頭アドレス
        size_t offset;                  //!< 現在のオフセット

        //! 初期化用コンストラクタ
        DynamicBuffer() noexcept
            : buffer(nullptr)
            , offset(0)
        {}
    };

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      動的バッファの取得
     *  \return     動的バッファ
     */
     /* ------------------------------------------------------------------------- */
    constexpr DynamicBuffer &GetVertexBuffer() noexcept
    {
        return _vertexBuffer;
    }


    ID3D11Buffer* GetConstantBuffer(size_t size) noexcept;

    bool SetRenderTarget(ID3D11RenderTargetView *renderTargetView, float width, float height) noexcept;

    bool ResizeClientBuffer(float width, float height) noexcept;

    DXGI_FORMAT GetMainRenderTargetPixelFormat() const noexcept;
    Vector2 GetMainRenderTargetSize() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      平行投影行列の取得
     *  \return     平行投影行列
     */
    /* ------------------------------------------------------------------------- */
    ID3D11Buffer *GetMainOrthogonalMatrixBuffer() const noexcept
    {
        return _mainOrthogonalMatrixBuffer;
    }

private:
    bool InitializeOrthogonalMatrixBuffer() noexcept;
    bool InitializeVertexBuffer() noexcept;
    bool InitializeConstantBuffer() noexcept;

    bool UpdateViewport(float width, float height) noexcept;
    bool UpdateOrthogonalMatrix() noexcept;

    static void OnEventPostFrameUpdate(void *callbackArg, void *notifyArg) noexcept;
    static void OnEventChangeClientSize(void *callbackArg, void *notifyArg) noexcept;

    D3D_DRIVER_TYPE                     _driverType;			    //!< ドライバタイプ
    D3D_FEATURE_LEVEL                   _featureLevel;			    //!< 機能レベル
    CComPtr<ID3D11Device>               _d3dDevice;			        //!< D3Dデバイス
    CComPtr<ID3D11Device1>              _d3dDevice1;			    //!< D3Dデバイス 11.1 以降
    CComPtr<ID3D11DeviceContext>        _immediateContext;		    //!< コンテキスト
    CComPtr<ID3D11DeviceContext1>       _immediateContext1;	        //!< コンテキスト 11.1 以降
    CComPtr<IDXGISwapChain>             _swapChain;			        //!< スワップチェーン
    CComPtr<IDXGISwapChain1>            _swapChain1;			    //!< スワップチェーン 11.1 以降
    CComPtr<ID3D11RenderTargetView>     _mainRenderTargetView;	    //!< メインの描画先
    CComPtr<ID3D11Texture2D>            _mainTexture;               //!< メインの描画先のテクスチャ
    CComPtr<ID3D11Buffer>               _mainOrthogonalMatrixBuffer;//!< メインの描画先の平行投影行列
    DynamicBuffer			            _vertexBuffer;			    //!< 動的頂点バッファ

    struct ConstBuffer
    {
        size_t size;
        CComPtr<ID3D11Buffer> buffer;

        ConstBuffer()
            : size(0)
            , buffer(nullptr)
        {}

        ConstBuffer(size_t inSize)
            : size(inSize)
            , buffer(nullptr)
        {}
    };

    std::vector<ConstBuffer>  _constBufferArray;
    ID3D11RenderTargetView *_currentRenderTarget;

    Event::Handle _eventPostFrameUpdate;
    Event::Handle _eventChangeClientSize;
};
}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
#endif	// INCGUARD_MGL_D3D11_DEVICE_H_1614578619

// vim: et ts=4 sw=4 sts=4
