// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_random_xorshift.h
 *  \brief      MGL 乱数生成
 *  \date       Since: July 22, 2021. 12:03:00 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RANDOM_H_1626922980
#define INCGUARD_MGL_RANDOM_H_1626922980

#include <cstdint>

namespace MGL::Random
{
//! XorShift乱数ジェネレータ
class XorShift
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  seed    シード値
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr XorShift(uint32_t seed = 0) noexcept
        : _seed(seed)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      乱数の取得
     *  \return     乱数
     */
    /* ------------------------------------------------------------------------- */
    constexpr uint32_t Get() noexcept
    {
        _seed = _seed ^ (_seed << 13);
        _seed = _seed ^ (_seed >> 17);
        _seed = _seed ^ (_seed << 5);

        return _seed;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      乱数の取得（float版）
     *  \return     乱数（0.0f - 1.0f）
     */
    /* ------------------------------------------------------------------------- */
    constexpr float GetFloat() noexcept
    {
        return static_cast<float>(static_cast<double>(Get()) / static_cast<double>(0xFFFFFFFF));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *   \brief      種の取得
     *   \return     現在の種
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr uint32_t GetSeed() const noexcept
    {
        return _seed;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *   \brief      種の設定
     *   \param[in]  seed    設定する種
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetSeed(uint32_t seed) noexcept
    {
        _seed = seed;
    }

private:
    uint32_t _seed;
};
}    // namespace MGL::Random

#endif    // INCGUARD_MGL_RANDOM_H_1626922980

// vim: et ts=4 sw=4 sts=4
