// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_win32_window.h
 *  \brief      MGL Win32用ウィンドウ
 *  \date       Since: March 1, 2021. 12:04:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_WIN32_WINDOW_H_1614567872
#define INCGUARD_MGL_WIN32_WINDOW_H_1614567872

#include <mgl/mgl_environment.h>
#if defined(MGL_TARGET_WIN32)

#include <Windows.h>
#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event_handle.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Win32
{
//! Win32用ウィンドウクラス
class Window final : public MGL::SharedSingleton<Window>
{
public:

    //! ウィンドウ初期化用記述子
    struct Descriptor
    {
        int width{1280};                    //!< ウィンドウの幅
        int height{720};                    //!< ウィンドウの高さ
        LPCWSTR windowTitle{L"MGL-App"};    //!< ウィンドウタイトル
        LPCWSTR className{nullptr};         //!< クラス名（nullptr指定でウィンドウタイトルを使用）
        HICON icon{nullptr};                //!< ウィンドウアイコン
        HICON smallIcon{nullptr};           //!< ウィンドウの小アイコン
    };

    static STL::unique_ptr<Window> &GetInstanceRef() noexcept;

    bool Initialize(HINSTANCE hInstance, int nCmdShow, WNDPROC windowProc, const Descriptor &descriptor) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウハンドラの取得
     */
    /* ------------------------------------------------------------------------- */
    constexpr HWND GetWindowHandler() noexcept
    {
        return _hWnd;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウの幅を取得
     *  \return     ウィンドウの幅
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr int GetWidth() const noexcept
    {
        return _width;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウの高さを取得
     *  \return     ウィンドウの高さ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr int GetHeight() const noexcept
    {
        return _height;
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションインスタンスのハンドルを取得
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr HINSTANCE GetAppInstanceHandle() const noexcept
    {
        return _hInstance;
    }

    void SetWindowTitle(const char *title) noexcept;
    bool Resize(int width, int height) noexcept;
    [[nodiscard]] float GetLogicalScaleFactor() const noexcept;

    void Update() noexcept;

private:
    HINSTANCE _hInstance{nullptr};
    HWND _hWnd{nullptr};
    int _width{0};
    int _height{0};
    float _scaleFactor{1.0f};
};
}   // namespace MGL::Win32

#endif  // MGL_TARGET_WIN32
#endif	// INCGUARD_MGL_WIN32_WINDOW_H_1614567872

// vim: et ts=4 sw=4 sts=4
