// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_version.h
 *  \brief      MGL バージョン情報
 *  \date       Since: November 28, 2020. 18:23:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_VERSION_H_1606555397
#define INCGUARD_MGL_VERSION_H_1606555397

#include <cstdint>

//! MGLバージョン生成マクロ
#define MGL_MAKE_VERSION(major, minor, patch) \
    ((major) * 10000000 + (minor) * 10000 + (patch))

// NOLINTBEGIN(modernize-macro-to-enum)
// Note: マクロのバージョン情報はプリプロセッサが判別するためのものなので列挙型にできない。
#define MGL_MAJOR_VERSION 1     //!< メジャーバージョン
#define MGL_MINOR_VERSION 1     //!< マイナーバージョン
#define MGL_PATCH_VERSION 16    //!< パッチバージョン
// NOLINTEND(modernize-macro-to-enum)

//! 現在のMGLのバージョンを表すマクロ
#define MGL_CURRENT_VERSION \
    MGL_MAKE_VERSION(MGL_MAJOR_VERSION, MGL_MINOR_VERSION, MGL_PATCH_VERSION)

namespace MGL
{
constexpr uint32_t kMajorVersion = MGL_MAJOR_VERSION;    //!< メジャーバージョン
constexpr uint32_t kMinorVersion = MGL_MINOR_VERSION;    //!< マイナーバージョン
constexpr uint32_t kPatchVersion = MGL_PATCH_VERSION;    //!< パッチバージョン

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バージョン値の生成
 *  \param[in]  major   メジャーバージョン
 *  \param[in]  minor   マイナーバージョン
 *  \param[in]  patch   パッチバージョン
 *  \return     バージョン値
 */
/* ------------------------------------------------------------------------- */
constexpr uint32_t MakeVersion(uint32_t major, uint32_t minor, uint32_t patch) noexcept
{
    return (major * 10000000) + (minor * 10000) + patch;
}

//! 現在のバージョン
constexpr uint32_t kCurrentVersion = MakeVersion(kMajorVersion, kMinorVersion, kPatchVersion);

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      特定のバージョン以降かを取得
 *  \param[in]  major   メジャーバージョン
 *  \param[in]  minor   マイナーバージョン
 *  \param[in]  patch   パッチバージョン
 *  \retval     true    現在のバージョンが指定したバージョン以降である
 *  \retval     false   現在のバージョンは指定したバージョンよりも古い
 */
/* ------------------------------------------------------------------------- */
constexpr bool IsALater(uint32_t major, uint32_t minor = 0, uint32_t patch = 0) noexcept
{
    return MakeVersion(major, minor, patch) >= kCurrentVersion;
}
}    // namespace MGL

#endif    // INCGUARD_MGL_VERSION_H_1606555397

// vim: et ts=4 sw=4 sts=4
