// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_utility.h
 *  \brief      MGL メモリユーティリティ
 *  \date       Since: July 11, 2025. 18:48:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MEMORY_UTILITY_H_1752227311
#define INCGUARD_MGL_MEMORY_UTILITY_H_1752227311

#include <cstdint>
#include <cstdlib>
#include <cstring>
#include <type_traits>

namespace MGL::Memory
{
//! クリアモード
enum class ClearMode : uint8_t
{
    Auto,    //!< 自動
    Zero,    //!< ゼロ初期化
    None,    //!< 初期化しない
};

namespace Utility
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      バッファを配列用に初期化
 *  \tparam     ValueType   配列の要素の型
 *  \param[in]  arrayTop    配列の先頭アドレス
 *  \param[in]  elementSize 配列の要素数
 *  \param[in]  mode        クリアモード
 */
/* ------------------------------------------------------------------------- */
template <class ValueType>
ValueType *InitializeArrayBuffer(void *arrayTop, size_t elementSize, ClearMode mode) noexcept
{
    // クラスかつデフォルトの場合は何もしない
    if constexpr (std::is_class_v<ValueType>)
    {
        if (mode == ClearMode::Auto)
        {
            mode = ClearMode::None;
        }
    }
    // それ以外の型はゼロクリア
    else
    {
        mode = ClearMode::Zero;
    }

    // メモリ領域のクリア
    if (mode == ClearMode::Zero)
    {
        memset(arrayTop, 0, sizeof(ValueType) * elementSize);
    }

    return new (arrayTop) ValueType[elementSize];
}
}    // namespace Utility
}    // namespace MGL::Memory

#endif    // INCGUARD_MGL_MEMORY_UTILITY_H_1752227311

// vim: et ts=4 sw=4 sts=4

