// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_fixed_size_allocator.h
 *  \brief      MGL 固定サイズアロケータ
 *  \date       Since: May 22, 2022. 18:20:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MEMORY_FIXED_SIZE_ALLOCATOR_H_1653211251
#define INCGUARD_MGL_MEMORY_FIXED_SIZE_ALLOCATOR_H_1653211251

#include <memory>
#include <mutex>

namespace MGL::Memory
{
//! 固定サイズアロケータクラス
class FixedSizeAllocator
{
public:
    //! メモリブロック情報
    struct Block
    {
        void *address;    //!< メモリブロックのアドレス
        Block *next;      //!< 次のメモリブロック
    };

    //! アロケータの種類を判別するためのタグ
    //  Note: デフォルトアロケータと絡んでいるため、これ単体で変更しない事！
    enum class Tag : int8_t
    {
        Fixed0 = 0,    //!< インデックス0の固定サイズアロケータ
        Fixed1 = 1,    //!< インデックス1の固定サイズアロケータ
        Fixed2 = 2,    //!< インデックス2の固定サイズアロケータ
        Fixed3 = 3,    //!< インデックス3の固定サイズアロケータ

        None = -1,     //!< 未アロケート
        Extra = -2,    //!< 外部アロケータを使用
    };

    //! ヘッダ情報
    struct Header
    {
        Tag tag;              //!< タグ
        int8_t unused;        //!< 未使用
        uint16_t usedSize;    //!< ブロック内の使用領域（ヘッダを除く）

        union
        {
            Block *block;          //!< 使用したメモリブロック（固定サイズアロケータを使用した場合）
            size_t requestSize;    //!< 要求サイズ（外部アロケータを使用した場合）
        };
    };
    static_assert(sizeof(Header) <= 16);    // ヘッダサイズは16バイト以下でなければならない

    bool Initialize(Tag tag, size_t blockSize, size_t blockCount) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      代替アロケータを設定
     *  \param[in]  altAllocator    設定する代替アロケータ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetAltAllocator(FixedSizeAllocator *altAllocator) noexcept
    {
        _altAllocator = altAllocator;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ブロックサイズを取得
     *  \return     ブロックサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetBlockSize() const noexcept
    {
        return _blockSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      確保可能な最大数を取得
     *  \return     確保可能な最大数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetCapacity() const noexcept
    {
        return _blockCount;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      利用可能なブロック数を取得
     *  \return     利用可能なブロック数
     */
    /* ------------------------------------------------------------------------- */
    size_t GetFreeCount() noexcept
    {
        const std::scoped_lock lock(_mutex);
        return _freeCount;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      利用中のブロック数を取得
     *  \return     利用中のブロック数
     */
    /* ------------------------------------------------------------------------- */
    size_t GetUsedCount() noexcept
    {
        return GetCapacity() - GetFreeCount();
    }

    void *Allocate(size_t size) noexcept;
    void Deallocate(void *buffer) noexcept;

private:
    Tag _tag{Tag::None};
    size_t _blockSize{0};
    size_t _blockCount{0};
    std::unique_ptr<std::byte[]> _pool{nullptr};
    size_t _poolSize{0};
    std::unique_ptr<Block[]> _blockArray{nullptr};
    Block *_topBlock{nullptr};
    size_t _freeCount{0};
    FixedSizeAllocator *_altAllocator{nullptr};

    std::mutex _mutex;
};
}    // namespace MGL::Memory

#endif    // INCGUARD_MGL_MEMORY_FIXED_SIZE_ALLOCATOR_H_1653211251

// vim: et ts=4 sw=4 sts=4
