// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_default_allocator.h
 *  \brief      MGL デフォルトのメモリアロケータ
 *  \date       Since: April 29, 2022. 20:24:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MEMORY_DEFAULT_ALLOCATOR_H_1651231444
#define INCGUARD_MGL_MEMORY_DEFAULT_ALLOCATOR_H_1651231444

#include <array>
#include <atomic>
#include <cstdio>
#include <cstdlib>

#include <mgl/memory/mgl_memory_allocator.h>
#include <mgl/memory/mgl_memory_fixed_size_allocator.h>

namespace MGL::Memory
{
//! デフォルトのメモリアロケータ
class DefaultAllocator : public Allocator
{
public:
    //! デフォルトアロケータの設定
    struct Configuration
    {
        //! 固定サイズアロケータの最小サイズ（2のn乗の値である必要がある）
        size_t fixedSizeAllocatorMinimumBlockSize{128};    // 最小128バイト

        //! 固定サイズアロケータのメモリプールのサイズ（最小サイズの8倍より大きくなければならない）
        size_t fixedSizeAllocatorMemoryPoolSize{static_cast<size_t>(1) * 1024 * 1024};    // メモリプールは1MiB

        //! 外部アロケータの指定（nullptr で malloc()を使用）
        void *(*allocator)(size_t size){nullptr};    // アロケータに malloc()を使用

        //! 外部デアロケータの指定（nullptr で free()を使用）
        void (*deallocator)(void *buffer){nullptr};    // デアロケータに free()を使用
                                                       //
        //! 外部リアロケータの指定（nullptr で realloc()を使用）
        void *(*reallocator)(void *buffer, size_t size){nullptr};    // リアロケータに realloc()を使用
    };

    DefaultAllocator() noexcept;

    //! このアロケータを表す値
    static constexpr AllocatorType kAllocatorType = MakeAllocatorType("MGL-DefaultAllocator");

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このアロケータを表す値を取得
     *  \return     このアロケータを表す値
     *  \note       MakeAllocatorType()によって生成された値を推奨
     */
    /* ------------------------------------------------------------------------- */
    AllocatorType GetType() const noexcept override
    {
        return kAllocatorType;
    }

    bool Initialize(const Configuration &config) noexcept;

    bool Initialize() noexcept override;
    bool IsInitialized() const noexcept override;
    [[nodiscard]] void *Allocate(size_t size) noexcept override;
    void Deallocate(void *buffer) noexcept override;
    [[nodiscard]] void *Reallocate(void *buffer, size_t newSize) noexcept override;

    bool GetSizeInfo(size_t &dest, uint32_t key, uint32_t arg) noexcept override;

private:
    FixedSizeAllocator *GetFixedSizeAllocator(const FixedSizeAllocator::Header &header) noexcept;

    bool _isInitialized{false};

    // 固定メモリアロケータの数。アルゴリズムに関わるので変更不可。
    static constexpr size_t kFixedSizeAllocatorCount = 4;

    Configuration _config;
    std::array<FixedSizeAllocator, kFixedSizeAllocatorCount> _fixedSizeAllocatorArray;
    size_t _fixedSizeBlockMax{0};
    uint8_t _shiftBit{0};

    // デバッグが無効ならこれらは利用しない
#if !defined(MGL_MEMORY_DEBUG_ENABLE)
    std::atomic<size_t> _systemUsedCount{0};
    std::atomic<size_t> _systemUsedSize{0};
#endif
};

}    // namespace MGL::Memory

#endif    // INCGUARD_MGL_MEMORY_DEFAULT_ALLOCATOR_H_1651231444

// vim: et ts=4 sw=4 sts=4
