// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_allocator.h
 *  \brief      MGL メモリアロケータ
 *  \date       Since: April 29, 2022. 20:21:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MEMORY_ALLOCATOR_H_1651231310
#define INCGUARD_MGL_MEMORY_ALLOCATOR_H_1651231310

#include <cstddef>
#include <cstdint>

#include <mgl/hash/mgl_hash_fnv1a.h>

namespace MGL::Memory
{

//! アロケータを判別するための型
enum class AllocatorType : uint32_t;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      AllocatorTypeを生成するための定数式
 *  \param[in]  key     生成に使用する文字列
 *  \return     生成されたアロケータタイプ
 */
/* ------------------------------------------------------------------------- */
constexpr AllocatorType MakeAllocatorType(const char *key) noexcept
{
    return static_cast<AllocatorType>(Hash::FNV1a(key));
}

//! 無効なアロケータを表す値
constexpr AllocatorType kInvalidAllocatorType = MakeAllocatorType("MGL-InvalidAllocator");

//! メモリアロケータインターフェース
class Allocator
{
public:
    virtual ~Allocator() = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このアロケータを表す値を取得
     *  \return     このアロケータを表す値
     *  \note       MakeAllocatorType()によって生成された値を推奨
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual AllocatorType GetType() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Initialize() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化状態を取得
     *  \retval     true    初期化済み
     *  \retval     false   未初期化
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsInitialized() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アロケート
     *  \param[in]  size    アロケートサイズ
     *  \return     アロケートしたメモリのアドレス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual void *Allocate(size_t size) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デアロケート
     *  \param[in]  buffer  デアロケートするメモリアドレス
     */
    /* ------------------------------------------------------------------------- */
    virtual void Deallocate(void *buffer) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リアロケート
     *  \param[in]  buffer  リアロケートするメモリアドレス
     *  \param[in]  newSize 新たに確保するメモリサイズ
     *  \return     アロケートしたメモリのアドレス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual void *Reallocate(void *buffer, size_t newSize) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズ情報を取得
     *  \param[out] dest    取得したサイズの格納先
     *  \param[in]  key     取得するサイズの種類を表すキー．内容は実装先依存
     *  \param[in]  arg     取得の際に使用する引数
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool GetSizeInfo(size_t &dest, uint32_t key, uint32_t arg) noexcept = 0;
};
}    // namespace MGL::Memory

#endif    // INCGUARD_MGL_MEMORY_ALLOCATOR_H_1651231310

// vim: et ts=4 sw=4 sts=4
