// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_vector2.h
 *  \brief      MGL 2Dベクトル
 *  \date       Since: November 29, 2020. 15:28:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_VECTOR2_H_1606631290
#define INCGUARD_MGL_VECTOR2_H_1606631290

#include <algorithm>
#include <cmath>

#include <mgl/math/mgl_math_defs.h>

namespace MGL
{
//! 2Dベクトル
struct Vector2
{
    float x;    //!< X成分
    float y;    //!< Y成分

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  ゼロ初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2() noexcept
        : x(0.0f)
        , y(0.0f)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値を指定して初期化
     *  \param[in]  inX     X成分
     *  \param[in]  inY     Y成分
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2(float inX, float inY) noexcept
        : x(inX)
        , y(inY)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      単項加算演算子
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2 &operator+=(const Vector2 &rhs) noexcept
    {
        this->x += rhs.x;
        this->y += rhs.y;

        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      単項減算演算子
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2 &operator-=(const Vector2 &rhs) noexcept
    {
        this->x -= rhs.x;
        this->y -= rhs.y;

        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      単項乗算演算子
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2 &operator*=(float rhs) noexcept
    {
        this->x *= rhs;
        this->y *= rhs;

        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      単項除算演算子
     */
    /* ------------------------------------------------------------------------- */
    constexpr Vector2 &operator/=(float rhs) noexcept
    {
        this->x /= rhs;
        this->y /= rhs;

        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値がほぼゼロかを調べる
     *  \retval     true    ほぼゼロ
     *  \retval     false   ほぼゼロでない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsZero() const noexcept
    {
        return ((std::fabs(x) <= Math::kEpsilon) && (std::fabs(y) <= Math::kEpsilon));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      長さを求める
     *  \return     長さ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float Length() const noexcept
    {
        return std::sqrt((x * x) + (y * y));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ターゲットまでの距離を求める
     *  \param[in]  target    ターゲット
     *  \return     ターゲットまでの距離
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float Distance(const Vector2 &target) const noexcept
    {
        return Vector2(x - target.x, y - target.y).Length();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      外積を求める
     *  \param[in]  rhs     右辺
     *  \return     外積
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float Cross(const Vector2 &rhs) const noexcept
    {
        return (x * rhs.y) - (rhs.x * y);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      外積を求める
     *  \param[in]  lhs     左辺
     *  \param[in]  rhs     右辺
     *  \return     外積
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] static float Cross(const Vector2 &lhs, const Vector2 &rhs) noexcept
    {
        return (lhs.x * rhs.y) - (rhs.x * lhs.y);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      内積を求める
     *  \param[in]  rhs     右辺
     *  \return     内積
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float Dot(const Vector2 &rhs) const noexcept
    {
        return (x * rhs.x) + (y * rhs.y);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      内積を求める
     *  \param[in]  lhs     左辺
     *  \param[in]  rhs     右辺
     *  \return     内積
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] static float Dot(const Vector2 &lhs, const Vector2 &rhs) noexcept
    {
        return (lhs.x * rhs.x) + (lhs.y * rhs.y);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      2点間の角度を求める
     *  \param[in]  target  ターゲットの点
     *  \return     このベクトルからターゲット方向への角度
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetAngle(const Vector2 &target) const noexcept
    {
        const auto length = Length() * target.Length();
        if ((length == 0.0f) || (length == -0.0f))
        {
            return 0.0f;
        }

        const auto dot = Dot(target);
        if ((dot == 0.0f) || (dot == -0.0f))
        {
            return Math::kHalfPi;
        }

        const float theta = std::clamp(dot / length, -1.0f, 1.0f);
        return std::acos(theta);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ベクトルを正規化する
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Normalize() noexcept
    {
        if ((x == 0.0f) || (x == -0.0f))
        {
            if ((y == 0.0f) || (y == -0.0f))
            {
                return false;
            }
        }

        const auto length = Length();
        x /= length;
        y /= length;

        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ベクトルを回転
     *  \param[in]  radian      回転角度
     */
    /* ------------------------------------------------------------------------- */
    void Rotate(float radian) noexcept
    {
        const auto srcX = x;
        x = (srcX * std::cos(radian)) - (y * std::sin(radian));
        y = (srcX * std::sin(radian)) + (y * std::cos(radian));
    }
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Vector2の加算
 *  \param[in]  lhs     左辺
 *  \param[in]  rhs     右辺
 *  \return     加算結果
 */
/* ------------------------------------------------------------------------- */
constexpr Vector2 operator+(const Vector2 &lhs, const Vector2 &rhs) noexcept
{
    return {lhs.x + rhs.x, lhs.y + rhs.y};
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Vector2の減算
 *  \param[in]  lhs     左辺
 *  \param[in]  rhs     右辺
 *  \return     減算結果
 */
/* ------------------------------------------------------------------------- */
constexpr Vector2 operator-(const Vector2 &lhs, const Vector2 &rhs) noexcept
{
    return {lhs.x - rhs.x, lhs.y - rhs.y};
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Vector2の乗算
 *  \param[in]  lhs     左辺
 *  \param[in]  rhs     右辺
 *  \return     乗算結果
 */
/* ------------------------------------------------------------------------- */
constexpr Vector2 operator*(const Vector2 &lhs, float rhs) noexcept
{
    return {lhs.x * rhs, lhs.y * rhs};
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Vector2の除算
 *  \param[in]  lhs     左辺
 *  \param[in]  rhs     右辺
 *  \return     除算結果
 */
/* ------------------------------------------------------------------------- */
constexpr Vector2 operator/(const Vector2 &lhs, float rhs) noexcept
{
    return {lhs.x / rhs, lhs.y / rhs};
}
}    // namespace MGL

#endif    // INCGUARD_MGL_VECTOR2_H_1606631290

// vim: et ts=4 sw=4 sts=4
