// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_rectangle.h
 *  \brief      MGL 矩形
 *  \date       Since: November 29, 2020. 15:34:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RECTANGLE_H_1606631656
#define INCGUARD_MGL_RECTANGLE_H_1606631656

#include <mgl/math/mgl_vector2.h>

namespace MGL
{
//! 矩形
struct Rectangle
{
    float x;         //!< X座標
    float y;         //!< Y座標
    float width;     //!< 幅
    float height;    //!< 高さ

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  ゼロ初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr Rectangle() noexcept
        : x(0.0f)
        , y(0.0f)
        , width(0.0f)
        , height(0.0f)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したパラメータで初期化
     *  \param[in]  inX         位置のX座標
     *  \param[in]  inY         位置のY座標
     *  \param[in]  inWidth     サイズの幅
     *  \param[in]  inHeight    サイズの高さ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Rectangle(float inX, float inY, float inWidth, float inHeight) noexcept
        : x(inX)
        , y(inY)
        , width(inWidth)
        , height(inHeight)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      位置とサイズで初期化
     *  \param[in]  position    位置
     *  \param[in]  size        サイズ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Rectangle(const Vector2 &position, const Vector2 &size) noexcept
        : x(position.x)
        , y(position.y)
        , width(size.x)
        , height(size.y)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      位置を設定
     *  \param[in]  position    位置
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPosition(const Vector2 &position) noexcept
    {
        x = position.x;
        y = position.y;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      位置を設定
     *  \param[in]  inX     X座標
     *  \param[in]  inY     Y座標
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPosition(float inX, float inY) noexcept
    {
        x = inX;
        y = inY;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズを設定
     *  \param[in]  size    サイズ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetSize(const Vector2 &size) noexcept
    {
        width = size.x;
        height = size.y;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズを設定
     *  \param[in]  inWidth     幅
     *  \param[in]  inHeight    高さ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetSize(float inWidth, float inHeight) noexcept
    {
        width = inWidth;
        height = inHeight;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      位置を取得
     *  \return     位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Vector2 GetPosition() const noexcept
    {
        return {x, y};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      サイズを取得
     *  \return     サイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Vector2 GetSize() const noexcept
    {
        return {width, height};
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した位置が矩形の内側にあるかを取得
     *  \param[in]  point   位置
     *  \retval     true    位置が矩形の内側にある
     *  \retval     false   位置は矩形の外側にある
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnclosed(const Vector2 &point) const noexcept
    {
        if ((x <= point.x) && (y <= point.y))
        {
            if (((x + width) >= point.x) && ((y + height) >= point.y))
            {
                return true;
            }
        }

        return false;
    }
};
}    // namespace MGL

#endif    // INCGUARD_MGL_RECTANGLE_H_1606631656

// vim: et ts=4 sw=4 sts=4
