// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_matrix4x4.h
 *  \brief      MGL 4x4行列
 *  \date       Since: March 2, 2021. 13:02:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MATRIX4X4_H_1614657758
#define INCGUARD_MGL_MATRIX4X4_H_1614657758

#include <mgl/math/mgl_vector4.h>
#include <mgl/mgl_environment.h>

namespace MGL
{
//! 単位行列を生成するための配列
constexpr float kIdentityMatrix4x4Array[16] =
{
    1.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 1.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 1.0f, 0.0f,
    0.0f, 0.0f, 0.0f, 1.0f
};

//! 4x4行列
struct Matrix4x4
{
    Vector4 column[4];    //!< 行列を表す4つのカラム

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゼロ初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr Matrix4x4() noexcept
        : column()
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      Vector4の配列で初期化
     *  \param[in]  vectorArray     初期化するVector4配列
     */
    /* ------------------------------------------------------------------------- */
    constexpr Matrix4x4(const Vector4 vectorArray[4]) noexcept
    {
        column[0] = vectorArray[0];
        column[1] = vectorArray[1];
        column[2] = vectorArray[2];
        column[3] = vectorArray[3];
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      floatの配列で初期化
     *  \param[in]  array     初期化するfloat配列
     */
    /* ------------------------------------------------------------------------- */
    constexpr Matrix4x4(const float array[16]) noexcept
    {
        column[0] = Vector4(array[0], array[1], array[2], array[3]);
        column[1] = Vector4(array[4], array[5], array[6], array[7]);
        column[2] = Vector4(array[8], array[9], array[10], array[11]);
        column[3] = Vector4(array[12], array[13], array[14], array[15]);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      単位行列化
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Identity() noexcept
    {
        *this = Matrix4x4(kIdentityMatrix4x4Array);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値の配列を取得
     */
    /* ------------------------------------------------------------------------- */
    float *GetArray() noexcept
    {
        return reinterpret_cast<float *>(column);
    }
};

// サイズチェック
static_assert(sizeof(Matrix4x4) == 16 * sizeof(float), "[MGL] Matrix4x4 is not float[16]");

//! 単位行列の定数
constexpr Matrix4x4 kIdentityMatrix4x4 = Matrix4x4(kIdentityMatrix4x4Array);

}    // namespace MGL

#endif    // INCGUARD_MGL_MATRIX4X4_H_1614657758

// vim: et ts=4 sw=4 sts=4
