// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_leaderboard_server_delegate.h
 *  \brief      MGL リーダーボードサーバのデリゲート
 *  \date       Since: September 4, 2021. 1:36:49 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_LEADERBOARD_SERVER_DELEGATE_H_1630687009
#define INCGUARD_MGL_LEADERBOARD_SERVER_DELEGATE_H_1630687009

#include <mgl/leaderboard/mgl_leaderboard_defs.h>

namespace MGL::Leaderboard
{
//! MGL リーダーボードサーバのデリゲートクラス
class ServerDelegate
{
public:
    virtual ~ServerDelegate() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Initialize() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリデータを取得
     *  \param[in]  identifier      リーダーボードの識別子
     *  \param[in]  startRank       取得する最初の順位
     *  \param[in]  requestSize     取得するエントリ数
     *  \param[in]  playerScope     取得するプレイヤーのスコープ
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool FetchEntryData(Identifier identifier, uint32_t startRank, uint32_t requestSize, PlayerScope playerScope) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      プレイヤーの順位を中心にエントリデータを取得
     *  \param[in]  identifier      リーダーボードの識別子
     *  \param[in]  requestSize     取得するエントリ数
     *  \param[in]  playerScope     取得するプレイヤーのスコープ
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool FetchEntryDataAroundUser(Identifier identifier, uint32_t requestSize, PlayerScope playerScope) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータのダウンロード
     *  \param[in]  clipID          クリップデータの識別子
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool DownloadClipData(uint64_t clipID) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータの送信
     *  \param[in]  identifier      リーダーボードの識別子
     *  \param[in]  clipData        送信するクリップデータ
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool UploadClipData(Identifier identifier, const ClipData &clipData) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スコアの送信
     *  \param[in]  identifier      リーダーボードの識別子
     *  \param[in]  score           送信するスコア
     *  \param[in]  options         送信するオプションデータ
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool ReportScore(Identifier identifier, uint64_t score, const STL::vector<uint32_t> &options) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリ取得結果を取得
     *  \return     取得結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const FetchResults &GetFetchedData() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スコアの送信結果を取得
     *  \return     送信結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const ReportResults &GetReportResults() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ダウンロードしたクリップデータを取得
     *  \return     クリップデータ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual const ClipData &GetClipData() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータのアップロード結果を取得
     *  \return     クリップデータのエラータイプ．成功している場合はErrorType::None
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual ClipData::ErrorType GetClipDataUploadResults() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理中かどうかを取得
     *  \retval     true    処理中
     *  \retval     false   処理中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsProcessing() const noexcept = 0;
};
}    // namespace MGL::Leaderboard

#endif    // INCGUARD_MGL_LEADERBOARD_SERVER_DELEGATE_H_1630687009

// vim: et ts=4 sw=4 sts=4
