// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_leaderboard_server.h
 *  \brief      MGL リーダーボードサーバ
 *  \date       Since: September 4, 2021. 1:27:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_LEADERBOARD_SERVER_H_1630686448
#define INCGUARD_MGL_LEADERBOARD_SERVER_H_1630686448

#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/leaderboard/mgl_leaderboard_server_delegate.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Leaderboard
{
//! MGL リーダーボードサーバ
class Server final : public MGL::SharedSingleton<Server>
{
public:
    //! ステート
    enum class State : uint8_t
    {
        NotReady,            //!< 準備できていない
        Ready,               //!< 準備完了
        FetchEntryData,      //!< エントリ取得中
        ReportScore,         //!< スコア送信中
        UploadClipData,      //!< クリップデータをアップロード中
        DownloadClipData,    //!< クリップデータをダウンロード中
    };

    static STL::unique_ptr<Server> &GetInstanceRef() noexcept;

    Server() noexcept;

    bool Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept;

    bool ReportScore(Identifier identifier, uint64_t score, const OptionDataArray &options) noexcept;
    bool FetchEntryData(Identifier identifier, uint32_t startRank, uint32_t requestSize, PlayerScope playerScope) noexcept;
    bool FetchEntryDataAroundUser(Identifier identifier, uint32_t requestSize, PlayerScope playerScope) noexcept;
    bool UploadClipData(Identifier identifier, const ClipData &clipData) noexcept;
    bool DownloadClipData(uint64_t clipID) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリ取得結果を取得
     *  \return     取得結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const FetchResults &GetFetchedData() const noexcept
    {
        return _delegate->GetFetchedData();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      送信結果を取得
     *  \return     送信結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const ReportResults &GetReportResults() const noexcept
    {
        return _delegate->GetReportResults();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ダウンロードしたクリップデータを取得
     *  \return     クリップデータ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const ClipData &GetClipData() const noexcept
    {
        return _delegate->GetClipData();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータのアップロード結果を取得
     *  \return     クリップデータのエラータイプ．成功している場合はErrorType::None
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] ClipData::ErrorType GetClipDataUploadResults() const noexcept
    {
        return _delegate->GetClipDataUploadResults();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理中かどうかを取得
     *  \retval     true    処理中
     *  \retval     false   処理中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsProcessing() const noexcept
    {
        return _state != State::Ready;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在のステートを取得
     *  \return     現在のステート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr State GetState() const noexcept
    {
        return _state;
    }

private:
    static void OnPreFrameUpdate(void *callbackArg, void *notifyArg) noexcept;

    STL::unique_ptr<ServerDelegate> _delegate;
    State _state{State::Ready};

    Event::Handle _eventPreFrameUpdate;
};
}    // namespace MGL::Leaderboard

#endif    // INCGUARD_MGL_LEADERBOARD_SERVER_H_1630686448

// vim: et ts=4 sw=4 sts=4
