// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_leaderboard_clipdata.h
 *  \brief      MGL リーダーボードのクリップデータ
 *  \date       Since: September 7, 2021. 6:17:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_LEADERBOARD_CLIPDATA_H_1630963048
#define INCGUARD_MGL_LEADERBOARD_CLIPDATA_H_1630963048

#include <mgl/file/mgl_file_path_view.h>
#include <mgl/stl/mgl_stl_containers.h>

namespace MGL::Leaderboard
{
//! クリップデータ
class ClipData
{
public:
    //! クリップデータの種類
    enum class DataType : uint8_t
    {
        None,          //!< なし
        File,          //!< ファイル
        MemoryData,    //!< メモリ上のデータ
        Invalid        //!< 不正なデータ
    };

    //! エラータイプ
    enum class ErrorType : uint8_t
    {
        None,                //!< エラーなし
        FileAccessFailed,    //!< ファイルアクセスに失敗
        ConnectionFailed,    //!< サーバとの通信に失敗
        DataNotFound,        //!< クリップデータが見つからなかった
    };

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      空データのコンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    ClipData() noexcept
        : _dataType(DataType::None)
        , _errorType(ErrorType::None)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルパスと関連付けるクリップデータのコンストラクタ
     *  \param[in]  filePath  クリップデータのファイルのパス
     */
    /* ------------------------------------------------------------------------- */
    ClipData(const File::PathView &filePath) noexcept
        : _dataType(DataType::File)
        , _filePath(filePath)
        , _errorType(ErrorType::None)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      メモリ上のデータと関連付けるクリップデータのコンストラクタ
     *  \param[in]  data  クリップデータ
     */
    /* ------------------------------------------------------------------------- */
    ClipData(const STL::vector<std::byte> &data) noexcept
        : _dataType(DataType::MemoryData)
        , _data(data)
        , _errorType(ErrorType::None)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      データタイプの取得
     *  \return     データタイプ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr DataType GetDataType() const noexcept
    {
        return _dataType;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータがファイルかを取得
     *  \retval     true    クリップデータはファイル
     *  \retval     false   ファイルではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsFile() const noexcept
    {
        return _dataType == DataType::File;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      クリップデータがメモリ上のデータかを取得
     *  \retval     true    クリップデータはメモリ上のデータ
     *  \retval     false   メモリ上のデータではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsMemoryData() const noexcept
    {
        return _dataType == DataType::MemoryData;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーが発生しているかを取得
     *  \retval     true    発生している
     *  \retval     false   発生していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasError() const noexcept
    {
        return _dataType == DataType::Invalid;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルパスを取得
     *  \return     ファイルパス．クリップデータがファイルでない場合は空のパス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const File::Path &GetFilePath() const noexcept
    {
        return _filePath;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      データを取得
     *  \return     データの配列
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const STL::vector<std::byte> &GetData() const noexcept
    {
        return _data;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーの種類を取得
     *  \return     発生しているエラー．発生していない場合はErrorType::None
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ErrorType GetErrorType() const noexcept
    {
        return HasError() ? _errorType : ErrorType::None;
    }

private:
    DataType _dataType;              //!< クリップデータのタイプ
    File::Path _filePath;            //!< アップロードするファイルのパス
    STL::vector<std::byte> _data;    //!< アップロードするデータ
    ErrorType _errorType;            //!< エラータイプ
};
}    // namespace MGL::Leaderboard


#endif    // INCGUARD_MGL_LEADERBOARD_CLIPDATA_H_1630963048

// vim: et ts=4 sw=4 sts=4
