// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_leaderboard.h
 *  \brief      MGL リーダーボード
 *  \date       Since: September 8, 2021. 2:56:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_LEADERBOARD_H_1631037391
#define INCGUARD_MGL_LEADERBOARD_H_1631037391

#include <mgl/leaderboard/mgl_leaderboard_server.h>

namespace MGL::Leaderboard
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リーダーボードサーバの生成
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
template <class DelegateClass, class... Args>
inline bool CreateServer(Args... args) noexcept
{
    if (Server::HasInstance())
    {
        return false;
    }

    auto &server = Server::CreateInstance();

    STL::unique_ptr<ServerDelegate> delegate = STL::make_unique<DelegateClass>(args...);

    return server.Initialize(delegate);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サーバの有効状態を取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsAvailableServer() noexcept
{
    return Server::HasInstance();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スコアの送信
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  score           送信するスコア
 *  \param[in]  options         送信するオプションデータ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool ReportScore(Identifier identifier, uint64_t score, const OptionDataArray &options = OptionDataArray()) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().ReportScore(identifier, score, options) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エントリデータを取得
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  startRank       取得する最初の順位
 *  \param[in]  requestSize     取得するエントリ数
 *  \param[in]  playerScope     取得するプレイヤーのスコープ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool FetchEntryData(Identifier identifier, uint32_t startRank, uint32_t requestSize, PlayerScope playerScope) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().FetchEntryData(identifier, startRank, requestSize, playerScope) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プレイヤーの順位を中心にエントリデータを取得
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  requestSize     取得するエントリ数
 *  \param[in]  playerScope     取得するプレイヤーのスコープ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool FetchEntryDataAroundUser(Identifier identifier, uint32_t requestSize, PlayerScope playerScope) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().FetchEntryDataAroundUser(identifier, requestSize, playerScope) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップデータの送信
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  clipData        送信するクリップデータ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool UploadClipData(Identifier identifier, const ClipData &clipData) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().UploadClipData(identifier, clipData) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップデータのダウンロード
 *  \param[in]  clipID          クリップデータの識別子
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool DownloadClipData(uint64_t clipID) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().DownloadClipData(clipID) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エントリ取得結果を取得
 *  \return     取得結果
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline const FetchResults &GetFetchedData() noexcept
{
    return Server::GetInstance().GetFetchedData();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      送信結果を取得
 *  \return     送信結果
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline const ReportResults &GetReportResults() noexcept
{
    return Server::GetInstance().GetReportResults();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ダウンロードしたクリップデータを取得
 *  \return     クリップデータ
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline const ClipData &GetClipData() noexcept
{
    return Server::GetInstance().GetClipData();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップデータのアップロード結果を取得
 *  \return     クリップデータのエラータイプ．成功している場合はErrorType::None
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline ClipData::ErrorType GetClipDataUploadResults() noexcept
{
    return IsAvailableServer() ? Server::GetInstance().GetClipDataUploadResults() : ClipData::ErrorType::None;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サーバが処理中かどうかを取得
 *  \retval     true    処理中
 *  \retval     false   処理中でない
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsProcessing() noexcept
{
    return IsAvailableServer() ? Server::GetInstance().IsProcessing() : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サーバが処理可能な状態かを取得
 *  \retval     true    処理中
 *  \retval     false   処理中でない
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsReady() noexcept
{
    return IsAvailableServer() ? !Server::GetInstance().IsProcessing() : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のサーバのステートを取得
 *  \return     現在のサーバのステート
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline Server::State GetServerState() noexcept
{
    return IsAvailableServer() ? Server::GetInstance().GetState() : Server::State::NotReady;
}
}    // namespace MGL::Leaderboard

#endif    // INCGUARD_MGL_LEADERBOARD_H_1631037391

// vim: et ts=4 sw=4 sts=4
