// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_touch_state.h
 *  \brief      MGL タッチ状態入力定義
 *  \date       Since: February 16, 2021. 3:16:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TOUCH_STATE_H_1613412992
#define INCGUARD_MGL_TOUCH_STATE_H_1613412992

#include <array>

#include <mgl/math/mgl_vector2.h>

namespace MGL::Input
{
//!< マルチタッチの最大認識数
constexpr size_t kMultiTouchCount = 5;

//! タッチ識別番号の型
enum class TouchID : uint32_t
{
    Invalid = 0xFFFFFFFF    //!< 無効値
};

//! TouchIDのプリインクリメントのオーバーロード
constexpr TouchID operator++(TouchID &rhs) noexcept
{
    rhs = TouchID{static_cast<std::underlying_type_t<TouchID>>(rhs) + 1};
    if (rhs >= TouchID::Invalid)
    {
        rhs = TouchID{0};
    }

    return rhs;
}

//! TouchIDのポストインクリメントのオーバーロード
constexpr TouchID operator++(TouchID &rhs, int) noexcept
{
    auto ret = rhs;
    rhs = TouchID{static_cast<std::underlying_type_t<TouchID>>(rhs) + 1};
    if (rhs >= TouchID::Invalid)
    {
        rhs = TouchID{0};
    }

    return ret;
}


//! タッチステート
struct TouchState
{
    bool isEnabled{false};                //!< 有効フラグ
    uint64_t deviceID{0};                 //!< デバイス側の識別番号
    TouchID touchID{TouchID::Invalid};    //!< タッチ識別番号
    uint32_t touchFrameCount{0};          //!< タッチフレーム数
    Vector2 position;                     //!< タッチ位置
    Vector2 prevPosition;                 //!< 前回のタッチ位置
    Vector2 deltaMove;                    //!< タッチ移動量
    Vector2 uvPosition;                   //!< タッチ位置（UV座標）
    Vector2 uvPrevPosition;               //!< 前回のタッチ位置（UV座標）
    Vector2 uvDeltaMove;                  //!< タッチ移動量（UV座標）


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチステートの更新処理
     *  \param[in]  newState    更新するステート
     */
    /* ------------------------------------------------------------------------- */
    void Update(const TouchState &newState) noexcept
    {
        isEnabled = newState.isEnabled;

        if (isEnabled)
        {
            deviceID = newState.deviceID;
            touchID = newState.touchID;
            touchFrameCount++;
            prevPosition = position;
            position = newState.position;
            deltaMove = position - prevPosition;
            uvPrevPosition = uvPosition;
            uvPosition = newState.uvPosition;
            uvDeltaMove = uvPosition - uvPrevPosition;
        }
        else
        {
            *this = TouchState();
        }
    }
};

//! タッチステートの配列
using TouchStateArray = std::array<TouchState, kMultiTouchCount>;

}    // namespace MGL::Input

#endif    // INCGUARD_MGL_TOUCH_STATE_H_1613412992

// vim: et ts=4 sw=4 sts=4
