// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_touch.h
 *  \brief      MGL タッチ入力取得
 *  \date       Since: February 16, 2021. 5:01:54 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_TOUCH_H_1613419314
#define INCGUARD_MGL_TOUCH_H_1613419314

#include <mgl/input/touch/mgl_touch_server.h>
#include <mgl/input/touch/mgl_touch_settings.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! MGL タッチ入力取得クラス
class Touch
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックス指定によるコンストラクタ
     *  \param[in]  index   インデックス（0 から kMultiTouchMax）
     */
    /* ------------------------------------------------------------------------- */
    Touch(size_t index = 0) noexcept
        : _state(TouchServer::GetInstance().GetState(index))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチ識別番号の指定によるコンストラクタ
     *  \param[in]  touchID     タッチ識別番号
     */
    /* ------------------------------------------------------------------------- */
    Touch(TouchID touchID) noexcept
        : _state(TouchServer::GetInstance().GetState(touchID))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      矩形指定によるコンストラクタ
     *  \param[in]  rectangle   判定する矩形
     *  \note       矩形の中にタッチステートが複数存在する場合，最もインデックスの小さいステートを扱う
     */
    /* ------------------------------------------------------------------------- */
    Touch(const Rectangle &rectangle) noexcept
        : _state(TouchServer::GetInstance().GetState(rectangle))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチサーバの有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    static bool IsAvailable() noexcept
    {
        return TouchServer::GetInstance().IsAvailable();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチの有効状態を取得
     *  \retval     true    有効（タッチされている）
     *  \retval     false   無効（タッチされていない）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabled() const noexcept
    {
        return _state.isEnabled;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチの識別番号を取得
     *  \return     識別番号．タッチされていない場合はkInvalidTouchID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr TouchID GetTouchID() const noexcept
    {
        return _state.touchID;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチされているフレーム数を取得
     *  \return     タッチされているフレーム数（1以上），タッチされていない場合は0
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr uint32_t GetTouchFrameCount() const noexcept
    {
        return _state.touchFrameCount;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチされた最初のフレームかを取得
     *  \retval     true    最初のタッチ
     *  \retval     false   継続（前のフレームからタッチされている）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsTouchedFirst() const noexcept
    {
        return _state.isEnabled && (_state.touchFrameCount == 1);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチ位置の取得
     *  \return     タッチ位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetPosition() const noexcept
    {
        return _state.position;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のタッチ位置の取得
     *  \return     前回のタッチ位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetPrevPosition() const noexcept
    {
        return _state.prevPosition;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のタッチ位置からの移動量を取得
     *  \return     前回のタッチ位置からの移動量
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetDeltaMove() const noexcept
    {
        return _state.deltaMove;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチ位置をUV座標で取得
     *  \return     タッチ位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetUVPosition() const noexcept
    {
        return _state.uvPosition;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のタッチ位置をUV座標で取得
     *  \return     前回のタッチ位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetUVPrevPosition() const noexcept
    {
        return _state.uvPrevPosition;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      前回のタッチ位置からの移動量をUV座標で取得
     *  \return     前回のタッチ位置からの移動量
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetUVDeltaMove() const noexcept
    {
        return _state.uvDeltaMove;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型にキャストした際に有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr operator bool() const noexcept
    {
        return IsEnabled();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態を否定演算子で取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!() const noexcept
    {
        return !static_cast<bool>(*this);
    }

private:
    const TouchState &_state;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_TOUCH_H_1613419314

// vim: et ts=4 sw=4 sts=4
