// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_state.h
 *  \brief      MGL マウス入力状態定義
 *  \date       Since: December 28, 2020. 17:57:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MOUSE_STATE_H_1609145867
#define INCGUARD_MGL_MOUSE_STATE_H_1609145867

#include <cstdint>

#include <mgl/common/mgl_bit.h>
#include <mgl/math/mgl_vector2.h>

namespace MGL::Input
{
//! マウスボタン
enum class MouseButton : uint8_t
{
    None = 0,       //!< 入力なし
    Left = 1,       //!< 左クリック
    Right = 2,      //!< 右クリック
    Middle = 3,     //!< 中央ボタン
    Other1 = 4,     //!< その他1（ボタン3）
    Other2 = 5,     //!< その他2（ボタン4）
    Other3 = 6,     //!< その他3（ボタン5）
    Other4 = 7,     //!< その他4（ボタン6）
    Other5 = 8,     //!< その他5（ボタン7）
    Other6 = 9,     //!< その他6（ボタン8）
    Other7 = 10,    //!< その他7（ボタン9）
    Other8 = 11,    //!< その他8（ボタン10）
};

//! マウスボタンの入力状態のビットフラグ型
using MouseButtonFlags = MGL::EnumBitFlags<MouseButton>;

//! マウスボタンのor演算の結果をフラグ型に変換するためのオペレータ（MouseButton | MouseButton）
constexpr MouseButtonFlags operator|(MouseButton lhs, MouseButton rhs) noexcept
{
    return MouseButtonFlags(lhs) | rhs;
}

//! マウスの全てのボタンを表す定数
constexpr MouseButtonFlags kMouseButtonAll = MouseButtonFlags(0xFFFFFFFE);

//! マウスの状態
struct MouseState
{
    Vector2 position;          //!< ポインタの位置
    Vector2 prevPosition;      //!< 前回のポインタの位置
    Vector2 deltaMove;         //!< ポインタの移動量
    Vector2 uvPosition;        //!< ポインタの位置（UV座標）
    Vector2 uvPrevPosition;    //!< 前回のポインタの位置（UV座標）
    Vector2 uvDeltaMove;       //!< ポインタの移動量（UV座標）
    Vector2 deltaWheel;        //!< ホイール移動量（未実装）

    MouseButtonFlags buttonFlags;        //!< ボタンの状態
    MouseButtonFlags prevButtonFlags;    //!< 前回のボタンの状態
};

//! カーソルモード
enum class CursorMode : uint8_t
{
    None,         //!< なし
    Pointer,      //!< ポインタ使用
    DeltaMove,    //!< 移動量のみ
};

//! 表示設定
enum class CursorVisibleMode : uint8_t
{
    Visible,          //!< 表示
    Invisible,        //!< 非表示
    AutoInvisible,    //!< 自動で非表示
};

}    // namespace MGL::Input

#endif    // INCGUARD_MGL_MOUSE_STATE_H_1609145867

// vim: et ts=4 sw=4 sts=4
