// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_server.h
 *  \brief      MGL マウスサーバ
 *  \date       Since: December 28, 2020. 17:37:59 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MOUSE_SERVER_H_1609144679
#define INCGUARD_MGL_MOUSE_SERVER_H_1609144679

#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/input/mgl_input_bounds.h>
#include <mgl/input/mouse/mgl_mouse_delegate.h>
#include <mgl/stl/mgl_stl_memory.h>


namespace MGL
{
class InitializerDelegate;
}

namespace MGL::Input
{
//! マウスサーバクラス
class MouseServer final : public SharedSingleton<MouseServer>
{
public:
    static STL::unique_ptr<MouseServer> &GetInstanceRef() noexcept;

    MouseServer() noexcept;

    bool Initialize(STL::unique_ptr<MouseDelegate> &delegate) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスサーバが有効かどうかを返す
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailable() const noexcept
    {
        return _delegate != nullptr;
    }

    [[nodiscard]] bool IsPressing(MouseButton button) const noexcept;
    [[nodiscard]] bool IsPressing(MouseButtonFlags buttonFlags) const noexcept;
    [[nodiscard]] bool IsPressingAny(MouseButtonFlags buttonFlags = kMouseButtonAll) const noexcept;
    [[nodiscard]] bool IsTriggered(MouseButton button) const noexcept;
    [[nodiscard]] bool IsReleased(MouseButton button) const noexcept;

    [[nodiscard]] const Vector2 &GetPosition() const noexcept;
    [[nodiscard]] const Vector2 &GetUVPosition() const noexcept;
    [[nodiscard]] const Vector2 &GetDeltaMove() const noexcept;
    [[nodiscard]] const Vector2 &GetUVDeltaMove() const noexcept;

    void SetBounds(const Rectangle &bounds) noexcept;
    void SetBounds(const Rectangle &bounds, const Vector2 &logicalSize) noexcept;

    bool SetCursorMode(CursorMode cursorMode) noexcept;
    [[nodiscard]] CursorMode GetCursorMode() const noexcept;

    void SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept;
    [[nodiscard]] CursorVisibleMode GetCursorVisibleMode() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスの接続状態を取得
     *  \retval     true    接続されている
     *  \retval     false   接続されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsConnecting() const noexcept
    {
        return (_delegate != nullptr) ? _delegate->IsConnecting() : false;
    }

private:
    static void OnEventPreFrameUpdate(void *callbackArg, void *notifyArg) noexcept;
    static void OnEventChangeClientSize(void *callbackArg, void *notifyArg) noexcept;

    void UpdatePosition() noexcept;

    STL::unique_ptr<MouseDelegate> _delegate;
    Event::Handle _eventPreFrameUpdate;
    Event::Handle _eventChangeClientSize;
    MouseState _state;

    InputBounds _bounds;
    Vector2 _clientSize;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_MOUSE_SERVER_H_1609144679

// vim: et ts=4 sw=4 sts=4
