// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_delegate_macos.h
 *  \brief      MGL macOS用マウスデリゲート
 *  \date       Since: February 22, 2021. 2:03:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MOUSE_DELEGATE_MACOS_H_1613927027
#define INCGUARD_MGL_MOUSE_DELEGATE_MACOS_H_1613927027

#include <mgl/mgl_environment.h>
#if defined(MGL_MOUSE_DELEGATE_ENABLE_MACOS)

#if defined(__OBJC__)
#include <AppKit/NSWindow.h>
#endif

#include <mgl/input/mouse/mgl_mouse_delegate.h>
#include <mgl/input/mouse/mgl_mouse_event.h>
#include <mgl/event/mgl_event.h>

namespace MGL::Input
{
#if !defined(__OBJC__)
class NSWindow;
#endif

//! イベントコールバックによるマウス入力デリゲートクラス
class MacOSMouseDelegate : public MouseDelegate
{
public:
    MacOSMouseDelegate() noexcept;
    
    void UpdateState(MouseState &state) noexcept override;
    void PostUpdatePosition(const MouseState &state) noexcept override;
    
    bool SetCursorMode(CursorMode cursorMode) noexcept override;
    [[nodiscard]] CursorMode GetCursorMode() const noexcept override;

    void SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept override;
    [[nodiscard]] CursorVisibleMode GetCursorVisibleMode() const noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスの接続状態を取得
     *  \retval     true    接続されている
     *  \retval     false   接続されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsConnecting() const noexcept override
    {
        return true;
    }

private:
    static void OnEventInputMouse(void *callbackArg, void *notifyArg) noexcept;
    static void OnEventShouldClearInput(void *callbackArg, void *notifyArg) noexcept;

    void UpdateButtonState(MouseButton button, bool isPressed) noexcept;
    void UpdatePosition(const Vector2 &position) noexcept;
    void UpdateDeltaMove(const Vector2 &deltaMove) noexcept;
    void UpdateDeltaWheel(const Vector2 &deltaWheel) noexcept;
    
    void UpdateCursorAutoErase(const MouseState &state, float elapsedTime) noexcept;
    void SetVisibleCursor(bool isVisible) noexcept;
    
    void MoveCursorToCenter(const NSWindow *window) noexcept;
    
    bool ShouldFixCursor(const NSWindow *window) const noexcept;
    
    Event::Handle _eventInputMouse;
    Event::Handle _eventShouldClearInput;

    MouseButtonFlags _buttonFlags;
    MouseButtonFlags _firstFrameButtonFlags;
    MouseButtonFlags _nextFrameReleaseFlags;
    
    Vector2 _position;
    Vector2 _deltaMove;
    Vector2 _deltaWheel;
    
    CursorMode _cursorMode;

    CursorVisibleMode _cursorVisibleMode;
    float _cursorDisplayTime;
    bool _isVisibleCursor;
};
}   // namespace MGL::Input

#endif  // MGL_MOUSE_DELEGATE_ENABLE_MACOS
#endif	// INCGUARD_MGL_MOUSE_DELEGATE_MACOS_H_1613927027

// vim: et ts=4 sw=4 sts=4
