// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse.h
 *  \brief      MGL マウス入力取得
 *  \date       Since: December 31, 2020. 17:55:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_MOUSE_H_1609404927
#define INCGUARD_MGL_MOUSE_H_1609404927

#include <mgl/input/mouse/mgl_mouse_server.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! MGL マウス入力取得クラス
class Mouse
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Mouse() noexcept
        : _server(MouseServer::GetInstance())
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスサーバを取得
     *  \return     マウスサーバ
     */
    /* ------------------------------------------------------------------------- */
    constexpr MouseServer &GetServer() noexcept
    {
        return _server;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスサーバが有効かどうかを返す
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailable() const noexcept
    {
        return _server.IsAvailable();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押されているかを取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押されている
     *  \retval     false   押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPressing(MouseButton button) const noexcept
    {
        return _server.IsPressing(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンが全て押されているかを取得
     *  \param[in]  buttonFlags     チェックするボタンフラグ
     *  \retval     true            押されている
     *  \retval     false           押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPressing(MouseButtonFlags buttonFlags) const noexcept
    {
        return _server.IsPressing(buttonFlags);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンのいずれかが押されているかを取得
     *  \param[in]  buttonFlags     チェックするボタンフラグ
     *  \retval     true            押されている
     *  \retval     false           押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPressingAny(MouseButtonFlags buttonFlags = kMouseButtonAll) const noexcept
    {
        return _server.IsPressingAny(buttonFlags);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押された瞬間である
     *  \retval     false   押された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsTriggered(MouseButton button) const noexcept
    {
        return _server.IsTriggered(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが離された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    離された瞬間である
     *  \retval     false   離された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsReleased(MouseButton button) const noexcept
    {
        return _server.IsReleased(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスポインタの位置を取得
     *  \return     マウスポインタの位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const Vector2 &GetPosition() const noexcept
    {
        return _server.GetPosition();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスポインタの位置をUV座標で取得
     *  \return     マウスポインタの位置
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const Vector2 &GetUVPosition() const noexcept
    {
        return _server.GetUVPosition();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスポインタの移動量を取得
     *  \return     マウスポインタの移動量
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const Vector2 &GetDeltaMove() const noexcept
    {
        return _server.GetDeltaMove();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスポインタの移動量をUV座標で取得
     *  \return     マウスポインタの移動量
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const Vector2 &GetUVDeltaMove() const noexcept
    {
        return _server.GetUVDeltaMove();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスの入力範囲を設定
     *  \param[in]  bounds      入力範囲
     */
    /* ------------------------------------------------------------------------- */
    void SetBounds(const Rectangle &bounds) noexcept
    {
        _server.SetBounds(bounds);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスの入力範囲を疑似サイズで設定
     *  \param[in]  bounds      入力範囲
     *  \param[in]  logicalSize 論理サイズ
     */
    /* ------------------------------------------------------------------------- */
    void SetBounds(const Rectangle &bounds, const Vector2 &logicalSize) noexcept
    {
        _server.SetBounds(bounds, logicalSize);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      カーソルモードを設定
     *  \param[in]  cursorMode  設定するカーソルモード
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    bool SetCursorMode(CursorMode cursorMode) noexcept
    {
        return _server.SetCursorMode(cursorMode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      カーソルモードを取得
     *  \return     現在のカーソルモード
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] CursorMode GetCursorMode() const noexcept
    {
        return _server.GetCursorMode();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      カーソルの表示モードを設定
     *  \param[in]  visibleMode 設定する表示モード
     */
    /* ------------------------------------------------------------------------- */
    void SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept
    {
        _server.SetCursorVisibleMode(visibleMode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      カーソルの表示モードを取得
     *  \return     現在のカーソルの表示モード
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] CursorVisibleMode GetCursorVisibleMode() const noexcept
    {
        return _server.GetCursorVisibleMode();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウスの接続状態を取得
     *  \retval     true    接続されている
     *  \retval     false   接続されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsConnecting() const noexcept
    {
        return _server.IsConnecting();
    }

private:
    MouseServer &_server;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_MOUSE_H_1609404927

// vim: et ts=4 sw=4 sts=4
