// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard_server.h
 *  \brief      MGL キーボードサーバ
 *  \date       Since: December 26, 2020. 21:23:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_KEYBOARD_SERVER_H_1608985408
#define INCGUARD_MGL_KEYBOARD_SERVER_H_1608985408

#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/input/keyboard/mgl_keyboard_delegate.h>
#include <mgl/input/mgl_input_repeat.h>

namespace MGL
{
class InitializerDelegate;
}

namespace MGL::Input
{
//! キーボードサーバ
class KeyboardServer final : public SharedSingleton<KeyboardServer>
{
public:
    static STL::unique_ptr<KeyboardServer> &GetInstanceRef() noexcept;

    KeyboardServer() noexcept;

    bool Initialize(STL::unique_ptr<KeyboardDelegate> &delegate) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボードサーバが有効かどうかを返す
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailable() const noexcept
    {
        return _delegate != nullptr;
    }

    [[nodiscard]] bool IsPressing(Keycode keycode) const noexcept;
    [[nodiscard]] bool IsTriggered(Keycode keycode) const noexcept;
    [[nodiscard]] bool IsReleased(Keycode keycode) const noexcept;
    [[nodiscard]] bool IsARepeat(Keycode keycode) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力を設定
     *  \param[in]  repeatSetting   リピート入力設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRepeatSetting(const RepeatSetting &repeatSetting) noexcept
    {
        _repeatSetting = repeatSetting;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定状態を取得
     *  \return     現在の設定状態
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const RepeatSetting &GetRepeatSetting() const noexcept
    {
        return _repeatSetting;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボードが接続されているかを取得
     *  \retval     true    接続されている
     *  \retval     false   接続されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsConnecting() const noexcept
    {
        return (_delegate != nullptr) ? _delegate->IsConnecting() : false;
    }

private:
    static void OnEventPreFrameUpdate(void *callbackArg, void *notifyArg);

    void UpdateModifierKeys() noexcept;
    void UpdateRepeatInput() noexcept;

    KeycodeArray _state;
    KeycodeArray _prevState;
    KeycodeArray _repeatState;

    STL::unique_ptr<KeyboardDelegate> _delegate;

    Event::Handle _eventPreFrameUpdate;

    RepeatSetting _repeatSetting;
    std::array<float, kKeycodeCount> _repeatTimerArray;
};
}    // namespace MGL::Input
#endif    // INCGUARD_MGL_KEYBOARD_SERVER_H_1608985408

// vim: et ts=4 sw=4 sts=4
