// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard.h
 *  \brief      MGL キーボード入力取得
 *  \date       Since: December 28, 2020. 15:17:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_KEYBOARD_H_1609136230
#define INCGUARD_MGL_KEYBOARD_H_1609136230

#include <mgl/input/keyboard/mgl_keyboard_server.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! キーボード入力取得クラス
class Keyboard
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    Keyboard() noexcept
        : _server(KeyboardServer::GetInstance())
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボードサーバの取得
     *  \return     キーボードサーバ
     */
    /* ------------------------------------------------------------------------- */
    constexpr KeyboardServer &GetServer() noexcept
    {
        return _server;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボードサーバが有効かどうかを返す
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAvailable() const noexcept
    {
        return _server.IsAvailable();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したキーが押されているかを取得
     *  \param[in]  keycode     キーコード
     *  \retval     true        押されている
     *  \retval     false       押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPressing(Keycode keycode) const noexcept
    {
        return _server.IsPressing(keycode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したキーが押された瞬間を取得
     *  \param[in]  keycode     キーコード
     *  \retval     true        押された瞬間である
     *  \retval     false       押された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsTriggered(Keycode keycode) const noexcept
    {
        return _server.IsTriggered(keycode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したキーが離された瞬間を取得
     *  \param[in]  keycode     キーコード
     *  \retval     true        離された瞬間である
     *  \retval     false       離された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsReleased(Keycode keycode) const noexcept
    {
        return _server.IsReleased(keycode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したキーのリピート入力状態を取得
     *  \param[in]  keycode     キーコード
     *  \retval     true        リピート入力あり
     *  \retval     false       リピート入力なし
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsARepeat(Keycode keycode) const noexcept
    {
        return _server.IsARepeat(keycode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力を設定
     *  \param[in]  repeatSetting   リピート入力設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRepeatSetting(const RepeatSetting &repeatSetting) noexcept
    {
        _server.SetRepeatSetting(repeatSetting);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定状態を取得
     *  \return     現在の設定状態
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const RepeatSetting &GetRepeatSetting() const noexcept
    {
        return _server.GetRepeatSetting();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボードが接続されているかを取得
     *  \retval     true    接続されている
     *  \retval     false   接続されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsConnecting() const noexcept
    {
        return _server.IsConnecting();
    }

private:
    KeyboardServer &_server;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_KEYBOARD_H_1609136230

// vim: et ts=4 sw=4 sts=4
