// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_state.h
 *  \brief      MGL ゲームパッドステート
 *  \date       Since: January 3, 2021. 15:13:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_STATE_H_1609654412
#define INCGUARD_MGL_GAMEPAD_STATE_H_1609654412

#include <array>
#include <atomic>

#include <mgl/input/gamepad/mgl_gamepad_defs.h>
#include <mgl/input/mgl_input_repeat.h>
#include <mgl/math/mgl_vector2.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! ゲームパッドステートクラス
class PadState
{
public:
    PadState() noexcept;

    void PreUpdate() noexcept;
    void PostUpdate(const RepeatSetting &repeatSetting) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      使用中フラグを設定
     *  \param[in]  isUsing     使用中フラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetUsing(bool isUsing) noexcept
    {
        _isUsing = isUsing;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      使用中フラグを取得
     *  \retval     true    使用中
     *  \retval     false   未使用
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsUsing() const noexcept
    {
        return _isUsing || IsEnabled();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アクティベート要求
     *  \param[in]  type        パッドの種類
     *  \param[in]  priority    パッドのプライオリティ
     *  \param[in]  deviceInfo  デバイス情報
     */
    /* ------------------------------------------------------------------------- */
    void RequestsActivate(PadType type, PadPriority priority, const PadDeviceInfo &deviceInfo) noexcept
    {
        _type = type;
        _priority = priority;
        _deviceInfo = deviceInfo;
        _shouldActivate = true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アクティベート解除要求
     */
    /* ------------------------------------------------------------------------- */
    void RequestsDeactivate() noexcept
    {
        _shouldDeactivate = true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      等価演算子のオーバーロード
     *  \param[in]  rhs     右辺
     *  \return     等価演算の結果
     */
    /* ------------------------------------------------------------------------- */
    bool operator==(const PadState &rhs) const noexcept
    {
        return _identifier == rhs._identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      不等価演算子のオーバーロード
     *  \param[in]  rhs     右辺
     *  \return     不等価演算の結果
     */
    /* ------------------------------------------------------------------------- */
    bool operator!=(const PadState &rhs) const noexcept
    {
        return _identifier != rhs._identifier;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このパッドステートが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabled() const noexcept
    {
        return _identifier != PadID::Invalid;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      プレイヤーインデックスの設定
     *  \param[in]  playerIndex     設定するプレイヤーインデックス
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPlayerIndex(int32_t playerIndex) noexcept
    {
        _playerIndex = playerIndex;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      プレイヤーインデックスの設定
     *  \return     設定済みのプレイヤーインデックス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr int32_t GetPlayerIndex() const noexcept
    {
        return _playerIndex;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドの種類を取得
     *  \return     設定済みのゲームパッドの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadType GetType() const noexcept
    {
        return _type;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー番号の設定
     *  \param[in]  entry   設定するエントリー番号
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetEntry(PadEntry entry) noexcept
    {
        if (entry < PadEntry::Reserve_Start)
        {
            _entry = entry;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー番号の取得
     *  \return     設定済みのエントリー番号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadEntry GetEntry() const noexcept
    {
        return _entry;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー済みかを取得
     *  \retval     true    エントリー済み
     *  \retval     false   エントリーしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasEntry() const noexcept
    {
        return IsEnabled() && (_entry < PadEntry::Reserve_Start);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンの設定
     *  \param[in]  button  設定するボタン
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetButton(PadButton button) noexcept
    {
        _buttonFlags.Set(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンの設定
     *  \param[in]  buttonFlags  設定するボタンフラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetButton(PadButtonFlags buttonFlags) noexcept
    {
        _buttonFlags |= buttonFlags;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      設定済みのボタンフラグの取得
     *  \return     設定済みのボタンフラグ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadButtonFlags GetButtonFlags() const noexcept
    {
        return _buttonFlags;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      左スティックの値を設定
     *  \param[in]  axisValue   左スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetLeftStick(const Vector2 &axisValue) noexcept
    {
        _leftStick = axisValue;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      右スティックの値を設定
     *  \param[in]  axisValue   右スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRightStick(const Vector2 &axisValue) noexcept
    {
        _rightStick = axisValue;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      左スティックの値を設定
     *  \return     左スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetLeftStick() const noexcept
    {
        return _leftStick;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      左スティックの値を設定（const版）
     *  \return     左スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Vector2 &GetLeftStick() noexcept
    {
        return _leftStick;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      右スティックの値を設定
     *  \return     右スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetRightStick() const noexcept
    {
        return _rightStick;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      右スティックの値を設定（const版）
     *  \return     右スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Vector2 &GetRightStick() noexcept
    {
        return _rightStick;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押されているかを取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押されている
     *  \retval     false   押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressing(PadButton button) const noexcept
    {
        return IsPressingAny(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンが全て押されているかを取得
     *  \param[in]  buttonFlags ボタンフラグ
     *  \retval     true        押されている
     *  \retval     false       押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressing(PadButtonFlags buttonFlags) const noexcept
    {
        return (_buttonFlags & buttonFlags) == buttonFlags;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンのいずれかが押されているかを取得
     *  \param[in]  buttonFlags ボタンフラグ
     *  \retval     true        押されている
     *  \retval     false       押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressingAny(PadButtonFlags buttonFlags = kGamepadButtonAll) const noexcept
    {
        return (_buttonFlags & buttonFlags).HasAny();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押された瞬間である
     *  \retval     false   押された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsTriggered(PadButton button) const noexcept
    {
        return IsPressing(button) && !_prevButtonFlags.Has(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが離された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    離された瞬間である
     *  \retval     false   離された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsReleased(PadButton button) const noexcept
    {
        return !IsPressing(button) && _prevButtonFlags.Has(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      いずれかのボタンの入力があるかを取得
     *  \retval     true    入力がある
     *  \retval     false   入力がない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasInputButton() const noexcept
    {
        return IsEnabled() && (_buttonFlags.HasAny() || _prevButtonFlags.HasAny());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    リピート入力がある
     *  \retval     false   リピート入力がない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsARepeat(PadButton button) const noexcept
    {
        return _repeatFlags.Has(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ポートレートモード（縦持ち）の設定
     *  \param[in]  isEnabled   有効フラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetPortraitMode(bool isEnabled) noexcept
    {
        _isPortraitMode = isEnabled;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ポートレートモード（縦持ち）の取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPortraitMode() const noexcept
    {
        return _isPortraitMode;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      プライオリティを取得
     *  \return     プライオリティ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadPriority GetPriority() const noexcept
    {
        return _priority;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      決定ボタンに割り当てるボタンを設定
     *  \param[in]  button  決定ボタンに割り当てるボタン
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetDecideButton(PadButton button) noexcept
    {
        _decideButton = button;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      決定ボタンに割り当てるボタンを取得
     *  \return     決定ボタンに割り当てるボタン
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadButton GetDecideButton() const noexcept
    {
        return _decideButton;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キャンセルボタンに割り当てるボタンを設定
     *  \param[in]  button  キャンセルボタンに割り当てるボタン
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetCancelButton(PadButton button) noexcept
    {
        _cancelButton = button;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キャンセルボタンに割り当てるボタンを取得
     *  \return     キャンセルボタンに割り当てるボタン
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadButton GetCancelButton() const noexcept
    {
        return _cancelButton;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デバイス情報を取得
     *  \return     デバイス情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const PadDeviceInfo &GetDeviceInfo() const noexcept
    {
        return _deviceInfo;
    }

    void ApplyLeftStickToDPad() noexcept;

    // コピーとムーブは禁止
    PadState(const PadState &) = delete;
    PadState &operator=(const PadState &) = delete;
    PadState(PadState &&) = delete;
    PadState &operator=(PadState &&) = delete;

private:
    void Activate() noexcept;
    void Deactivate() noexcept;

    void ApplyDecideCancelButton() noexcept;
    void UpdateRepeatFlags(const RepeatSetting &repeatSetting) noexcept;

    PadType _type{PadType::Disable};
    PadID _identifier{PadID::Invalid};
    PadDeviceInfo _deviceInfo;
    bool _isUsing{false};

    PadButtonFlags _buttonFlags;
    PadButtonFlags _prevButtonFlags;
    PadButtonFlags _repeatFlags;

    Vector2 _leftStick;
    Vector2 _rightStick;

    int32_t _playerIndex{-1};
    PadEntry _entry{PadEntry::NoEntry};

    PadPriority _priority{PadPriority::Low};

    PadButton _decideButton{PadButton::None};
    PadButton _cancelButton{PadButton::None};

    bool _isPortraitMode{false};

    std::array<float, PadButtonFlags::kSize> _repeatTimerArray;

    std::atomic<bool> _shouldActivate{false};
    std::atomic<bool> _shouldDeactivate{false};
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_GAMEPAD_STATE_H_1609654412

// vim: et ts=4 sw=4 sts=4
