// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_settings.h
 *  \brief      MGL ゲームパッドの設定
 *  \date       Since: February 8, 2022. 2:19:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_SETTINGS_H_1644254356
#define INCGUARD_MGL_GAMEPAD_SETTINGS_H_1644254356

#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! ゲームパッド設定クラス
class GamepadSettings
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    GamepadSettings() noexcept
        : _server(GamepadServer::GetInstance())
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定
     *  \param[in]  repeatSetting   リピート設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRepeatSetting(const RepeatSetting &repeatSetting) noexcept
    {
        _server.SetRepeatSetting(repeatSetting);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定状態を取得
     *  \return     現在のリピート入力の設定状態
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const RepeatSetting &GetRepeatSetting() const noexcept
    {
        return _server.GetRepeatSetting();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      決定ボタンとキャンセルボタンの割り当て
     *  \param[in]  type            パッドの種類
     *  \param[in]  decideButton    決定ボタン
     *  \param[in]  cancelButton    キャンセルボタン
     */
    /* ------------------------------------------------------------------------- */
    void SetDecideCancelButton(PadType type, PadButton decideButton, PadButton cancelButton) noexcept
    {
        _server.SetDecideCancelButton(type, decideButton, cancelButton);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      決定ボタンの割り当てを取得
     *  \param[in]  type    パッドの種類
     *  \return     決定ボタンに割り当てられているボタン
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] PadButton GetDecideButton(PadType type) const noexcept
    {
        return _server.GetDecideButton(type);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キャンセルボタンの割り当てを取得
     *  \param[in]  type    パッドの種類
     *  \return     キャンセルボタンに割り当てられているボタン
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] PadButton GetCancelButton(PadType type) const noexcept
    {
        return _server.GetCancelButton(type);
    }

private:
    GamepadServer &_server;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_GAMEPAD_SETTINGS_H_1644254356

// vim: et ts=4 sw=4 sts=4
