// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_server.h
 *  \brief      MGL ゲームパッドサーバ
 *  \date       Since: January 3, 2021. 15:11:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_SERVER_H_1609654269
#define INCGUARD_MGL_GAMEPAD_SERVER_H_1609654269

#include <array>
#include <mutex>
#include <optional>

#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/input/gamepad/mgl_gamepad_delegate.h>
#include <mgl/input/gamepad/mgl_gamepad_state.h>
#include <mgl/input/mgl_input_repeat.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL
{
class InitializerDelegate;
}

namespace MGL::Input
{
constexpr size_t kGamepadMax = 32;    //!< ゲームパッドの最大認識数

//! ゲームパッドサーバ
class GamepadServer final : public MGL::SharedSingleton<GamepadServer>
{
public:
    static STL::unique_ptr<GamepadServer> &GetInstanceRef() noexcept;

    GamepadServer() noexcept;

    bool AddDelegate(STL::unique_ptr<GamepadDelegate> delegate) noexcept;

    PadState *RequestsFreePadState() noexcept;
    void ReturnPadState(PadState *returnState) noexcept;
    size_t GetFreePadStateCount() const noexcept;

    PadEntry Entry(const PadState &entryState, PadEntry entry) noexcept;
    void Leave(const PadState &leaveState) noexcept;
    void LeaveAll() noexcept;

    const PadState &GetPadState(PadEntry entry = PadEntry::Auto) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定
     *  \param[in]  repeatSetting   リピート設定
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetRepeatSetting(const RepeatSetting &repeatSetting) noexcept
    {
        _repeatSetting = repeatSetting;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力の設定状態を取得
     *  \return     現在のリピート入力の設定状態
     */
    /* ------------------------------------------------------------------------- */
    constexpr const RepeatSetting &GetRepeatSetting() const noexcept
    {
        return _repeatSetting;
    }

    void SetDecideCancelButton(PadType type, PadButton decideButton, PadButton cancelButton) noexcept;

    PadButton GetDecideButton(PadType type, PadButton defaultButton = PadButton::None) const noexcept;
    PadButton GetCancelButton(PadType type, PadButton defaultButton = PadButton::None) const noexcept;

private:
    static void OnEventPreFrameUpdate(void *callbackArg, void *notifyArg) noexcept;

    std::optional<size_t> GetEntryIndex(PadEntry entry) const noexcept;
    void SetEntryIndex(PadEntry entry, int32_t index) noexcept;

    Event::Handle _eventPreFrameUpdate;

    STL::vector<STL::unique_ptr<GamepadDelegate>> _delegates;
    std::array<PadState, kGamepadMax> _padStates;

    std::array<int32_t, kGamepadEntryMax> _entryIndexes;

    bool _existsHighPriorityPad{false};

    PadState _invalidPadState;

    RepeatSetting _repeatSetting;

    struct TypeSettings
    {
        PadButton decideButton{PadButton::None};
        PadButton cancelButton{PadButton::None};
    };
    STL::unordered_map<PadType, TypeSettings> _typeSettings;

    std::recursive_mutex _mutexPadStates;
    std::recursive_mutex _mutexEntry;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_GAMEPAD_SERVER_H_1609654269

// vim: et ts=4 sw=4 sts=4
