// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_delegate_apple_gc.h
 *  \brief      MGL GameControllerフレームワークデリゲート
 *  \date       Since: September 24, 2022. 16:33:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_DELEGATE_APPLE_GC_H_1664004796
#define INCGUARD_MGL_GAMEPAD_DELEGATE_APPLE_GC_H_1664004796

#include <mgl/mgl_environment.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_APPLE_GC)

#include <array>
#include <mutex>

#if defined(__OBJC__)
#include <GameController/GameController.h>
#endif

#include <mgl/input/gamepad/mgl_gamepad_delegate.h>
#include <mgl/input/gamepad/mgl_gamepad_state.h>

namespace MGL::Input
{
// GameControllerフレームワークのクラスObj-Cのみ使用可能なので，C++では前方宣言しておく
#if !defined(__OBJC__)
namespace
{
class GCController;
};
#endif

//! MFiゲームパッドデリゲートクラス
class AppleGCGamepadDelegate : public GamepadDelegate
{
public:
    AppleGCGamepadDelegate(GamepadServer &server) noexcept;
    ~AppleGCGamepadDelegate() noexcept;

    virtual bool Initialize() noexcept override;
    virtual void UpdateState() noexcept override;

private:
    void InitializeAvailableFlags() noexcept;
    
    PadType GetPadType(GCController *controller) const noexcept;

    void OnConnect(GCController *controller) noexcept;
    void OnDisconnect(GCController *controller) noexcept;
    
    void UpdateMFiExtendedPadState(PadState &state, GCController *controller, bool isRequestedPause) noexcept;
    void UpdateMFiMicroPadState(PadState &state, GCController *controller, bool isRequestedPause) noexcept;
    void UpdateXboxOnePadState(PadState &state, GCController *controller) noexcept;
    void UpdateDualShock4PadState(PadState &state, GCController *controller) noexcept;
    void UpdateDualSensePadState(PadState &state, GCController *controller) noexcept;

#if !defined(__OBJC__)
    using id = void *;
#endif
    id _connectObserver;
    id _disconnectObserver;

#if defined(MGL_TARGET_TVOS)
    static constexpr size_t kGamepadMax = 3;  // tvOSのMFiゲームパッドの最大認識数
#else
    static constexpr size_t kGamepadMax = 4;  // macOS/iOSのMFiゲームパッドの最大認識数
#endif

    // ゲームパッドクラス
    struct Gamepad
    {
        PadState *state;                    // パッドステート
        GCController *controller;           // GameControllerフレームワークが管理するコントローラクラス
        std::atomic<bool> isRequestedPause; // ポーズ要求フラグ（古いバージョンのMFiゲームパッドのみで使用）
    };
    std::array<Gamepad, kGamepadMax> _gamepads;

    // 各機能の対応フラグ
    enum class AvailableFlag
    {
        MFiReadMenuButton,      // MFiゲームパッドのメニューボタンの直接読み取り
        MFiThumbstickButton,    // MFiゲームパッドのサムスティックボタンの読み取り
        MFiOptionsButton,       // MFiゲームパッドのオプションボタンの読み取り
        DualShock4Gamepad,      // DualShock4
        DualSenseGamepad,       // DualSense
        XboxOneGamepad,         // XboxOneコントローラ
        XboxShareButton,        // XBoxOneのシェアボタンの読み取り
    };
    static constexpr size_t kAvailableFlagSize = static_cast<size_t>(AvailableFlag::XboxShareButton) + 1;
    std::array<bool, kAvailableFlagSize> _availableFlags;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      各機能の対応状況を取得
     *  \retval     true    対応している
     *  \retval     false   対応していない
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool IsAvailable(AvailableFlag flag) const noexcept
    {
        return _availableFlags[static_cast<size_t>(flag)];
    }
    
    std::mutex _mutex;
};
}   // namespace MGL::Input

#endif  // MGL_GAMEPAD_DELEGATE_ENABLE_APPLE_GC
#endif	// INCGUARD_MGL_GAMEPAD_DELEGATE_APPLE_GC_H_1664004796

// vim: et ts=4 sw=4 sts=4
