// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_defs.h
 *  \brief      ゲームパッド関連各種定義
 *  \date       Since: September 24, 2022. 16:46:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_DEFS_H_1664005612
#define INCGUARD_MGL_GAMEPAD_DEFS_H_1664005612

#include <type_traits>
#include <string>

#include <mgl/common/mgl_bit.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::Input
{
//! ゲームパッドのIDを表す型
enum class PadID : uint32_t
{
    Invalid,                                                         //!< 無効値
    Any,                                                             //!< 入力のあるいずれかのパッドを表す値
    UniqueIDStart,                                                   //!< ユニークIDの開始値
    UniqueIDEnd = ~static_cast<std::underlying_type_t<PadID>>(0),    //!< ユニークIDの終了値
};

//! ゲームパッドのボタンの定義
enum class PadButton : uint8_t
{
    // 基本
    None        = 0,    //!< 入力なし
    Up          = 1,    //!< 上
    Down        = 2,    //!< 下
    Left        = 3,    //!< 左
    Right       = 4,    //!< 右
    AnalogUp    = 5,    //!< アナログ入力の上
    AnalogDown  = 6,    //!< アナログ入力の下
    AnalogLeft  = 7,    //!< アナログ入力の左
    AnalogRight = 8,    //!< アナログ入力の右
    Button01    = 9,    //!< ボタン1
    Button02    = 10,   //!< ボタン2
    Button03    = 11,   //!< ボタン3
    Button04    = 12,   //!< ボタン4
    Button05    = 13,   //!< ボタン5
    Button06    = 14,   //!< ボタン6
    Button07    = 15,   //!< ボタン7
    Button08    = 16,   //!< ボタン8
    Button09    = 17,   //!< ボタン9
    Button10    = 18,   //!< ボタン10
    Button11    = 19,   //!< ボタン11
    Button12    = 20,   //!< ボタン12
    Button13    = 21,   //!< ボタン13
    Button14    = 22,   //!< ボタン14
    Button15    = 23,   //!< ボタン15
    Button16    = 24,   //!< ボタン16
    Button17    = 25,   //!< ボタン17
    Button18    = 26,   //!< ボタン18
    Button19    = 27,   //!< ボタン19
    Button20    = 28,   //!< ボタン20
    Button21    = 29,   //!< ボタン21
    Cancel      = 30,   //!< キャンセル
    Decide      = 31,   //!< 決定

    // 共通の別名
    L1              = Button05,     //!< L1/LB
    R1              = Button06,     //!< R1/RB
    L2              = Button07,     //!< L2/LT
    R2              = Button08,     //!< R2/RT
    PrimaryMenu     = Button09,     //!< Pause/Start/Menu
    SecondaryMenu   = Button10,     //!< Select/Back/View
    Home            = Button11,     //!< ホームボタン
    L3              = Button12,     //!< L3/左サムスティックボタン
    R3              = Button13,     //!< R3/右サムスティックボタン
    Other1          = Button14,     //!< その他1
    Other2          = Button15,     //!< その他2
    Other3          = Button16,     //!< その他3
    Other4          = Button17,     //!< その他4
    Other5          = Button18,     //!< その他5
    Other6          = Button19,     //!< その他6
    Other7          = Button20,     //!< その他7
    Other8          = Button21,     //!< その他8

    // ダイヤモンド配置のボタン
    DiaDown     = Button01,         //!< ダイヤ配置の下ボタン
    DiaRight    = Button02,         //!< ダイヤ配置の右ボタン
    DiaLeft     = Button03,         //!< ダイヤ配置の左ボタン
    DiaUp       = Button04,         //!< ダイヤ配置の上ボタン

    // 任天堂系のボタン
    NintendoA   = DiaRight,         //!< 任天堂のAボタン
    NintendoB   = DiaDown,          //!< 任天堂のBボタン
    NintendoX   = DiaUp,            //!< 任天堂のXボタン
    NintendoY   = DiaLeft,          //!< 任天堂のYボタン

    // XInputのボタン
    XInputA         = DiaDown,          //!< XInputのAボタン
    XInputB         = DiaRight,         //!< XInputのBボタン
    XInputX         = DiaLeft,          //!< XInputのXボタン
    XInputY         = DiaUp,            //!< XInputのYボタン
    XInputMenu      = PrimaryMenu,      //!< XInputのMenu/Startボタン
    XInputView      = SecondaryMenu,    //!< XInputのView/Backボタン
    XInputShare     = Other1,           //!< XInputのShareボタン
    XInputPaddle1   = Other2,           //!< XInputの背面パドル1
    XInputPaddle2   = Other3,           //!< XInputの背面パドル2
    XInputPaddle3   = Other4,           //!< XInputの背面パドル3
    XInputPaddle4   = Other5,           //!< XInputの背面パドル4

    // DualShock/DualSenseのボタン
    DSCircle    = DiaRight,         //!< DualShock/DualSenseの◯ボタン
    DSCross     = DiaDown,          //!< DualShock/DualSenseの×ボタン
    DSTriangle  = DiaUp,            //!< DualShock/DualSenseの△ボタン
    DSSquare    = DiaLeft,          //!< DualShock/DualSenseの□ボタン
    DSOptions   = PrimaryMenu,      //!< DualShock/DualSenseのOptionsボタン
    DSShare     = SecondaryMenu,    //!< DualShock/DualSenseのShare/Createボタン
    DSTouchPad  = Other1,           //!< DualShock/DualSenseのタッチパッドボタン

    // MFi配置のボタン
    MFiA        = DiaDown,          //!< MFi配置のAボタン
    MFiB        = DiaRight,         //!< MFi配置のBボタン
    MFiX        = DiaLeft,          //!< MFi配置のXボタン
    MFiY        = DiaUp,            //!< MFi配置のYボタン

    // Apple Remote/Siri Remoteのボタン
    MFiTouchSurface =   Other1,     //!< タッチサーフェスのタッチ
};

//! ゲームパッドボタンの入力状態のビットフラグ型
using PadButtonFlags = EnumBitFlags<PadButton>;

//! ゲームパッドボタンのor演算の結果をフラグ型に変換するためのオペレータ（PadButton | PadButton）
constexpr PadButtonFlags operator|(PadButton lhs, PadButton rhs) noexcept
{
    return PadButtonFlags(lhs) | rhs;
}

//! ゲームパッドの全てのボタンを表す定数
constexpr PadButtonFlags kGamepadButtonAll = PadButtonFlags(0xFFFFFFFE);

//! ゲームパッドのボタンの最大数
constexpr size_t kGamepadButtonMax = 21;

//! ゲームパッドの種類
enum class PadType : uint8_t
{
    Disable,           //!< 無効/未接続
    MFiExtended,       //!< MFi拡張
    MFiMicro,          //!< AppleRemote/SiriRemote
    NintendoSwitch,    //!< Nintendo Switch
    DualShock4,        //!< DualShock4
    DualSense,         //!< DualSense
    XboxOne,           //!< Xbox One
    GenericHID,        //!< 汎用HID
    XInput,            //!< XInput
    DirectInput,       //!< DirectInput
    Other,             //!< その他
};

//! パッドのエントリータイプ
enum class PadEntry : uint8_t
{
    Player1 = 0,    //!< プレイヤー1
    Player2,        //!< プレイヤー2
    Player3,        //!< プレイヤー3
    Player4,        //!< プレイヤー4
    Player5,        //!< プレイヤー5
    Player6,        //!< プレイヤー6
    Player7,        //!< プレイヤー7
    Player8,        //!< プレイヤー8

    // 以下予約
    Reserve_Start,                             //!< 予約開始位置
    Reserve_MaxPlayerCount = Reserve_Start,    //!< エントリー可能なプレイヤーの最大数
    Reserve_NoEntry,                           //!< エントリーなしの指定
    Reserve_Any,                               //!< 全ての指定
    Reserve_Auto,                              //!< 自動指定

    // 特殊なプレイヤー指定
    NoEntry = Reserve_NoEntry,    //!< エントリーなしを取得
    Any = Reserve_Any,            //!< 全てを取得
    Auto = Reserve_Auto,          //!< エントリー済みを優先して取得し，無ければNoEntryと同じ
};

//! ゲームパッドの最大エントリー数
constexpr size_t kGamepadEntryMax = static_cast<size_t>(PadEntry::Reserve_MaxPlayerCount);

//! パッドのプライオリティ
enum class PadPriority : uint8_t
{
    Low,    //!< 低
    High    //!< 高
};

//! ゲームパッドのデバイス情報
struct PadDeviceInfo
{
    STL::string vendorName;     //!< 製造元の名前
    STL::string productName;    //!< 製品名
    uint16_t vendorID{};        //!< 製造元のID
    uint16_t productID{};       //!< 製品ID
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_GAMEPAD_DEFS_H_1664005612

// vim: et ts=4 sw=4 sts=4
