// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad.h
 *  \brief      MGL ゲームパッド
 *  \date       Since: January 8, 2021. 14:20:01 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_GAMEPAD_H_1610083201
#define INCGUARD_MGL_GAMEPAD_H_1610083201

#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/mgl_environment.h>

namespace MGL::Input
{
//! MGL ゲームパッド
class Gamepad
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  entry   エントリー番号．Any, Auto, NoEntry指定可能
     */
    /* ------------------------------------------------------------------------- */
    Gamepad(PadEntry entry = PadEntry::Auto) noexcept
        : _server(GamepadServer::GetInstance())
        , _state(_server.GetPadState(entry))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このゲームパッドが有効かどうかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsEnabled() const noexcept
    {
        return _state.IsEnabled();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      プレイヤーインデックスの設定
     *  \return     設定済みのプレイヤーインデックス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr int32_t GetPlayerIndex() const noexcept
    {
        return _state.GetPlayerIndex();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドの種類を取得
     *  \return     設定済みのゲームパッドの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadType GetType() const noexcept
    {
        return _state.GetType();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドの名前を取得
     *  \return     ゲームパッドの名前
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetName() const noexcept
    {
        return _state.GetDeviceInfo().productName.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドのエントリー
     *  \param[in]  entry       エントリー番号．Autoを指定すると空いている番号を自動で設定
     *  \return     エントリーされた番号．失敗した場合はNoEntry
     */
    /* ------------------------------------------------------------------------- */
    PadEntry Entry(PadEntry entry = PadEntry::Auto) noexcept
    {
        return _server.Entry(_state, entry);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドのエントリーの解除
     */
    /* ------------------------------------------------------------------------- */
    void Leave() noexcept
    {
        _server.Leave(_state);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー番号の取得
     *  \return     設定済みのエントリー番号
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr PadEntry GetEntry() const noexcept
    {
        return _state.GetEntry();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー番号を数値で取得
     *  \return     エントリー番号．未エントリーは0
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr uint8_t GetEntryNumber() const noexcept
    {
        return HasEntry() ? static_cast<uint8_t>(_state.GetEntry()) + 1 : 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エントリー済みかを取得
     *  \retval     true    エントリー済み
     *  \retval     false   エントリーしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasEntry() const noexcept
    {
        return _state.HasEntry();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      左スティックの値を設定
     *  \return     左スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetLeftStick() const noexcept
    {
        return _state.GetLeftStick();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      右スティックの値を設定
     *  \return     右スティックの軸の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Vector2 &GetRightStick() const noexcept
    {
        return _state.GetRightStick();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押されているかを取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押されている
     *  \retval     false   押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressing(PadButton button) const noexcept
    {
        return _state.IsPressing(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンが全て押されているかを取得
     *  \param[in]  buttonFlags ボタンフラグ
     *  \retval     true        押されている
     *  \retval     false       押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressing(PadButtonFlags buttonFlags) const noexcept
    {
        return _state.IsPressing(buttonFlags);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したボタンのいずれかが押されているかを取得
     *  \param[in]  buttonFlags ボタンフラグ
     *  \retval     true        押されている
     *  \retval     false       押されていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPressingAny(PadButtonFlags buttonFlags = kGamepadButtonAll) const noexcept
    {
        return _state.IsPressingAny(buttonFlags);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが押された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    押された瞬間である
     *  \retval     false   押された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsTriggered(PadButton button) const noexcept
    {
        return _state.IsTriggered(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボタンが離された瞬間を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    離された瞬間である
     *  \retval     false   離された瞬間ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsReleased(PadButton button) const noexcept
    {
        return _state.IsReleased(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      リピート入力を取得
     *  \param[in]  button  チェックするボタン
     *  \retval     true    リピート入力がある
     *  \retval     false   リピート入力がない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsARepeat(PadButton button) const noexcept
    {
        return _state.IsARepeat(button);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      パッドステートを取得
     *  \return     パッドステート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const PadState &GetState() const noexcept
    {
        return _state;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型にキャストした際に有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    explicit constexpr operator bool() const noexcept
    {
        return IsEnabled();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効状態を否定演算子で取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!() const noexcept
    {
        return !static_cast<bool>(*this);
    }

private:
    GamepadServer &_server;
    const PadState &_state;
};
}    // namespace MGL::Input

#endif    // INCGUARD_MGL_GAMEPAD_H_1610083201

// vim: et ts=4 sw=4 sts=4
