// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_iokit_hid_driver.h
 *  \brief      MGL IOKit HID ゲームパッドドライバ
 *  \date       Since: January 11, 2021. 17:49:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_IOKIT_HID_DRIVER_H_1610354956
#define INCGUARD_MGL_IOKIT_HID_DRIVER_H_1610354956

#include <mgl/mgl_environment.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID)

#include <cstdint>

#include <IOKit/hid/IOHIDManager.h>

#include <mgl/input/gamepad/mgl_gamepad_state.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Input::IOKitHID
{
//! MGL IOKit HID ゲームパッドドライバ
class GamepadDriver
{
public:
    virtual ~GamepadDriver() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化処理
     *  \param[in]  device  デバイス
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Initialize(IOHIDDeviceRef device) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ゲームパッドステートの更新処理
     *  \param[in,out]  state   ゲームパッドステート
     *  \param[in]      device  デバイス
     *  \retval         true    成功
     *  \retval         false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool UpdateState(MGL::Input::PadState &state, IOHIDDeviceRef device) noexcept = 0;
};


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ドライバが対応しているかをチェックして生成
 *  \tparam     DriverClass 生成するドライバクラス
 *  \param[in]  vendorID    デバイスのベンダID
 *  \param[in]  productID   デバイスのプロダクトID
 *  \return     対応している場合は生成したユニークポインタを，未対応の場合はnullptrを返す
 */
/* ------------------------------------------------------------------------- */
template <class DriverClass>
STL::unique_ptr<GamepadDriver> MakeGamepadDriver(int32_t vendorID, int32_t productID) noexcept
{
    if (DriverClass::IsManageable(vendorID, productID))
    {
        return STL::make_unique<DriverClass>();
    }

    return nullptr;
}

//! ドライバ生成関数の型
using MakeGamepadDriverFunction = STL::unique_ptr<GamepadDriver> (*)(int32_t vendorID, int32_t productID);

}    // namespace MGL::Input::IOKitHID
#endif    // MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID
#endif    // INCGUARD_MGL_IOKIT_HID_DRIVER_H_1610354956

// vim: et ts=4 sw=4 sts=4
