// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_iokit_hid_axis.h
 *  \brief      MGL IOKit HID用軸パラメータクラス
 *  \date       Since: January 14, 2021. 1:11:48 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_IOKIT_HID_AXIS_H_1610554308
#define INCGUARD_MGL_IOKIT_HID_AXIS_H_1610554308

#include <mgl/mgl_environment.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID)

#include <IOKit/hid/IOHIDManager.h>

namespace MGL::Input::IOKitHID
{
//! IOKit HID用軸パラメータ
class Axis
{
public:
    void SetParameter(IOHIDElementRef elemRef) noexcept;
    [[nodiscard]] float GetValue(int32_t value) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      軸の値がプラス方向を表しているかを取得
     *  \param[in]  value   デバイスから取得した値
     *  \retval     true    プラス方向である
     *  \retval     false   中央またはマイナス方向である
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPositive(int32_t value) const noexcept
    {
        return (_isAvailable && (static_cast<float>(value) > (_center + _threshold)));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      軸の値がマイナス方向を表しているかを取得
     *  \param[in]  value   デバイスから取得した値
     *  \retval     true    マイナス方向である
     *  \retval     false   中央またはプラス方向である
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsNegative(int32_t value) const noexcept
    {
        return (_isAvailable && (static_cast<float>(value) < (_center - _threshold)));
    }

private:
    bool _isAvailable{false};
    float _min{0.0f};
    float _max{0.0f};
    float _center{0.0f};
    float _threshold{0.0f};
};
}    // namespace MGL::Input::IOKitHID

#endif    // MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID
#endif    // INCGUARD_MGL_IOKIT_HID_AXIS_H_1610554308

// vim: et ts=4 sw=4 sts=4
