// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_initializer_delegate.h
 *  \brief      MGL イニシャライザデリゲート
 *  \date       Since: March 18, 2021. 14:57:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_INITIALIZER_DELEGATE_H_1616047024
#define INCGUARD_MGL_INITIALIZER_DELEGATE_H_1616047024

#include <mgl/audio/mgl_audio_defs.h>
#include <mgl/file/mgl_file_defs.h>
#include <mgl/mgl_environment.h>
#include <mgl/render/mgl_render_types.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/system/debug/mgl_debug_log_level.h>
#include <mgl/system/locale/mgl_locale_info.h>

namespace MGL
{
// レンダラ関連の前方宣言
namespace Render
{
class Renderer2DDelegate;
class TextureGenerator;
class TextureStorage;
}    // namespace Render

// オーディオ関連の前方宣言
namespace Audio
{
class Renderer;
}

// システム関連の前方宣言
namespace System
{
class ApplicationModuleInterface;
class ChronoModuleInterface;
class DebugModuleInterface;
class LocaleModuleInterface;
class WindowModuleInterface;
}    // namespace System

// 入力関連の前方宣言
namespace Input
{
class KeyboardDelegate;
class MouseDelegate;
class TouchDelegate;
class GamepadDelegate;
class GamepadServer;
}    // namespace Input

//! イニシャライザデリゲート
class InitializerDelegate
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    virtual ~InitializerDelegate() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オーディオの初期化モードを取得
     *  \return     オーディオの初期化モード
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Audio::InitializeMode GetAudioInitializeMode() const noexcept
    {
        return _audioInitMode;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オーディオの初期化タイプを設定
     *  \param[in]  mode    設定するオーディオの初期化タイプ
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetAudioInitializeMode(Audio::InitializeMode mode) noexcept
    {
        _audioInitMode = mode;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトのログレベルを取得
     *  \return     デフォルトのログレベル
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr System::LogLevel GetLogLevel() const noexcept
    {
        return _logLevel;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトのログレベルを設定
     *  \param[in]  logLevel    設定するログレベル
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetLogLevel(System::LogLevel logLevel) noexcept
    {
        _logLevel = logLevel;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      使用する言語の取得
     *  \return     使用する言語
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr System::Language GetLanguage() const noexcept
    {
        return _language;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      使用する言語の設定
     *  \param[in]  language    使用する言語
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetLanguage(System::Language language) noexcept
    {
        _language = language;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      2Dレンダラを生成
     *  \return     2Dレンダラ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Render::Renderer2DDelegate> MakeRenderer2D() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャジェネレータを生成
     *  \return     テクスチャジェネレータ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Render::TextureGenerator> MakeTextureGenerator() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      テクスチャローダーを生成
     */
    /* ------------------------------------------------------------------------- */
    virtual void MakeTextureLoader(Render::TextureStorage &textureStorage) const noexcept
    {
        (void)textureStorage;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オーディオレンダラの生成
     *  \return     オーディオレンダラ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Audio::Renderer> MakeAudioRenderer() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルシステム初期化後の処理
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual File::Result DidInitializeFileSystem() noexcept
    {
        return File::Error::NoOperation;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アプリケーションモジュールの生成
     *  \return     アプリケーションモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<System::ApplicationModuleInterface> MakeApplicationModule() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      日付と時刻のモジュールを生成
     *  \return     日付と時刻のモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<System::ChronoModuleInterface> MakeChronoModule() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デバッグモジュールを生成
     *  \return     デバッグモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<System::DebugModuleInterface> MakeDebugModule() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ロケール情報モジュールを生成
     *  \return     ロケール情報モジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<System::LocaleModuleInterface> MakeLocaleModule() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ウィンドウモジュールを生成
     *  \return     ウィンドウモジュール
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<System::WindowModuleInterface> MakeWindowModule() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      キーボード入力デリゲートの生成
     *  \return     キーボード入力デリゲート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Input::KeyboardDelegate> MakeKeyboardDelegate() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウス入力デリゲートの生成
     *  \return     マウス入力デリゲート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Input::MouseDelegate> MakeMouseDelegate() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      タッチ入力デリゲートの生成
     *  \return     タッチ入力デリゲート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual STL::unique_ptr<Input::TouchDelegate> MakeTouchDelegate() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ゲームパッドサーバ初期化後の処理
     *  \param[in]  server      ゲームパッドサーバ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool DidInitializeGamepadServer(Input::GamepadServer &server) const noexcept
    {
        (void)server;
        return true;
    }

private:
    Audio::InitializeMode _audioInitMode{Audio::InitializeMode::None};
    System::LogLevel _logLevel{System::LogLevel::AppTrace};

    System::Language _language{System::Language::Unknown};
};
}    // namespace MGL

#endif    // INCGUARD_MGL_INITIALIZER_DELEGATE_H_1616047024

// vim: et ts=4 sw=4 sts=4
