// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_initialize.h
 *  \brief      MGL 初期化と終了
 *  \date       Since: February 26, 2021. 17:54:07 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_INITIALIZE_H_1614329647
#define INCGUARD_MGL_INITIALIZE_H_1614329647

#include <mgl/initialize/mgl_initializer_delegate.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL
{
//! アプリケーションデリゲート生成関数の型
using MakeAppDelegateFunction = void (*)(STL::unique_ptr<ApplicationDelegate> &);

bool Initialize(InitializerDelegate &initializer, MakeAppDelegateFunction makeAppDelegate) noexcept;
void Terminate() noexcept;
bool IsInitialized() noexcept;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGL初期化処理
 *  \tparam     T               生成するアプリケーションデリゲートの型
 *  \param[in]  initializer     イニシャライザクラス
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
template <class T>
bool Initialize(InitializerDelegate &initializer) noexcept
{
    return Initialize(initializer, [](STL::unique_ptr<ApplicationDelegate> &appDelegate)
    {
        appDelegate = STL::make_unique<T>();
    });
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      NullプラットフォームでMGLを初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
inline bool NullInitialize() noexcept
{
    InitializerDelegate initializer;
    return Initialize(initializer, nullptr);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      NullプラットフォームでMGLを初期化
 *  \tparam     T       生成するアプリケーションデリゲートの型
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
template <class T>
bool NullInitialize() noexcept
{
    InitializerDelegate initializer;
    return Initialize(initializer, [](STL::unique_ptr<ApplicationDelegate> &appDelegate)
    {
        appDelegate = STL::make_unique<T>();
    });
}
}    // namespace MGL
#endif    // INCGUARD_MGL_INITIALIZE_H_1614329647

// vim: et ts=4 sw=4 sts=4
