// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_hash_fnv1a.h
 *  \brief      MGL FNV-1aハッシュ計算関数
 *  \date       Since: December 7, 2020. 11:44:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_HASH_FNV1A_H_1607309087
#define INCGUARD_MGL_HASH_FNV1A_H_1607309087

#include <cstdint>

namespace MGL::Hash
{
//! 32bitハッシュ値のデフォルト値
constexpr uint32_t kFNV1aDefaultValue32  = 0xA3F6C23E;
//! 32bitハッシュ値の生成に使用する素数
constexpr uint32_t kFNV1aPrime32         = 0x1000193;
//! 64bitハッシュ値のデフォルト値
constexpr uint64_t kFNV1aDefaultValue64  = 0xC18F36FE9A341BC5;
//! 64bitハッシュ値の生成に使用する素数
constexpr uint64_t kFNV1aPrime64         = 0x100000001B3;


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      FNV-1aアルゴリズムで文字列から32bitハッシュ値を取得
 *  \param[in]  str     文字列
 *  \param[in]  value   デフォルトの32bit値
 *  \return     32bitハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr uint32_t FNV1a(const char *str, const uint32_t value = Hash::kFNV1aDefaultValue32) noexcept
{
    if (str[0] == '\0')
    {
        return value;
    }
    
    return FNV1a(&str[1], (value ^ static_cast<uint32_t>(str[0])) * Hash::kFNV1aPrime32);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      FNV-1aアルゴリズムで文字列から64bitハッシュ値を取得
 *  \param[in]  str     文字列
 *  \param[in]  value   デフォルトの64bit値
 *  \return     64bitハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr uint64_t FNV1a64(const char *str, const uint64_t value = Hash::kFNV1aDefaultValue64) noexcept
{
    if (str[0] == '\0')
    {
        return value;
    }
    
    return FNV1a64(&str[1], (value ^ static_cast<uint64_t>(str[0])) * Hash::kFNV1aPrime64);
}
}   // namespace MGL::Hash
#endif	// INCGUARD_MGL_HASH_FNV1A_H_1607309087

// vim: et ts=4 sw=4 sts=4
